// Copyright (c) 2013, Matt Godbolt
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without 
// modification, are permitted provided that the following conditions are met:
// 
// Redistributions of source code must retain the above copyright notice, this 
// list of conditions and the following disclaimer.
// 
// Redistributions in binary form must reproduce the above copyright notice, 
// this list of conditions and the following disclaimer in the documentation 
// and/or other materials provided with the distribution.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
// SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
// CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
// POSSIBILITY OF SUCH DAMAGE.

#include "internal/HybiPacketDecoder.h"
#include "internal/LogStream.h"

#include <arpa/inet.h>

namespace seasocks {

HybiPacketDecoder::HybiPacketDecoder(Logger& logger, const std::vector<uint8_t>& buffer) :
    _logger(logger),
    _buffer(buffer),
    _messageStart(0) {
}

HybiPacketDecoder::MessageState HybiPacketDecoder::decodeNextMessage(std::vector<uint8_t>& messageOut) {
    if (_messageStart + 1 >= _buffer.size()) {
        return NoMessage;
    }
    if ((_buffer[_messageStart] & 0x80) == 0) {
        // FIN bit is not clear...
        // TODO: support
        LS_WARNING(&_logger, "Received hybi frame without FIN bit set - unsupported");
        return Error;
    }
    if ((_buffer[_messageStart] & (7<<4)) != 0) {
        LS_WARNING(&_logger, "Received hybi frame with reserved bits set - error");
        return Error;
    }
    auto opcode = _buffer[_messageStart] & 0xf;
    size_t payloadLength = _buffer[_messageStart + 1] & 0x7f;
    auto maskBit = _buffer[_messageStart + 1] & 0x80;
    auto ptr = _messageStart + 2;
    if (payloadLength == 126) {
        if (_buffer.size() < 4) { return NoMessage; }
        payloadLength = htons(*reinterpret_cast<const uint16_t*>(&_buffer[ptr]));
        ptr += 2;
    } else if (payloadLength == 127) {
        if (_buffer.size() < 10) { return NoMessage; }
        payloadLength = __bswap_64(*reinterpret_cast<const uint64_t*>(&_buffer[ptr]));
        ptr += 8;
    }
    uint32_t mask = 0;
    if (maskBit) {
        // MASK is set.
        if (_buffer.size() < ptr + 4) { return NoMessage; }
        mask = htonl(*reinterpret_cast<const uint32_t*>(&_buffer[ptr]));
        ptr += 4;
    }
    auto bytesLeftInBuffer = _buffer.size() - ptr;
    if (payloadLength > bytesLeftInBuffer) { return NoMessage; }

    messageOut.clear();
    messageOut.reserve(payloadLength);
    for (auto i = 0u; i < payloadLength; ++i) {
        auto byteShift = (3 - (i & 3)) * 8;
        messageOut.push_back(static_cast<char>((_buffer[ptr++] ^ (mask >> byteShift)) & 0xff));
    }
    _messageStart = ptr;
    switch (opcode) {
    default:
        LS_WARNING(&_logger, "Received hybi frame with unknown opcode " << opcode);
        return Error;
    case OPCODE_TEXT:
        return TextMessage;
    case OPCODE_BINARY:
        return BinaryMessage;
    case OPCODE_PING:
        return Ping;
    case OPCODE_CLOSE:
        return Close;
    }
}

size_t HybiPacketDecoder::numBytesDecoded() const {
    return _messageStart;
}

}
