(*:Name: SimpleHull` *)

(*:Title: SimpleHull *)

(*:Author: Tom Wickham-Jones*)

(*:Package Version: 1.0 *)

(*:Mathematica Version: 2.2 *)

(*:Summary:
	This package provides functions to perform a variety of
	two-dimensional geometric computations.
*)

(*:History:
	Created summer 1993 by Tom Wickham-Jones.

	This package is described in the book
	Mathematica Graphics: Techniques and Applications.
	Tom Wickham-Jones, TELOS/Springer-Verlag 1994.

*)


BeginPackage[ "ExtendGraphics`SimpleHull`"]


SimpleConvexHull::usage = 
	"SimpleConvexHull[ pts] will take a list of two-dimensional points 
	and return the vertices of the convex hull."


If[ Names[ "ExtendGraphics`Delaunay`ConvexHull"] === {},
	ToExpression[
		"ConvexHull::usage = 
	\"ConvexHull[ pts] will take a list of two-dimensional points 
	and return the vertices of the convex hull.\""]];
	
If[ Names[ "ExtendGraphics`Delaunay`ConvexHullPlot"] === {},
	ToExpression[
		"ConvexHullPlot::usage = 
	\"ConvexHullPlot[ pts] will plot the convex hull of a list of 
	two-dimensonal points.\""]];
	


Begin["`Private`"]

points

ConvexHullPlot[ pts_List /; 
			MatrixQ[ pts, NumberQ] && 
		        Length[ First[ pts]] === 2, opts___] :=
	Block[{hull},
		hull = ConvexHull[ pts] ;
		hull = Part[ pts, hull] ;
		hull = Line[ Append[ hull, First[ hull]]] ;
		Show[ Graphics[ {hull, AbsolutePointSize[ 3], 
				Map[ Point, pts]}, AspectRatio -> Automatic], opts]
		]
				

ConvexHull[ pts_List /; 
			MatrixQ[ pts, NumberQ] && 
		        Length[ First[ pts]] === 2] :=
    Block[{len, points, encode},
		len = Length[ pts] ;
    	points = Transpose[ {pts, Range[ len]}] ;
    	points = Sort[ points, 
					If[ #1[[1,1]] - #2[[1,1]] == 0, 
    					#1[[1,2]] < #2[[1,2]],
						#1[[1,1]] < #2[[1,1]]]&] ;
		{points, encode} = Transpose[ points] ;
		hull = Hull[ 1, Length[ points]] ;
		Part[ encode, hull]
	]

SimpleConvexHull[ pts_List /; 
			MatrixQ[ pts, NumberQ] && 
		        Length[ First[ pts]] === 2] :=
    Block[{len, points, encode},
		len = Length[ pts] ;
    	points = Transpose[ {pts, Range[ len]}] ;
    	points = Sort[ points, 
					If[ #1[[1,1]] - #2[[1,1]] == 0, 
    					#1[[1,2]] < #2[[1,2]],
						#1[[1,1]] < #2[[1,1]]]&] ;
		{points, encode} = Transpose[ points] ;
		hull = Hull[ 1, Length[ points]] ;
		Part[ encode, hull]
	]


Hull[ a_, b_] :=
    Block[{len},
    	len = b-a+1 ;
		If[ len > 5,
			MergeHull[ 
				Hull[ a, a+Floor[ len/2]-1],
				Hull[ a+Floor[ len/2], b]],
			SimpleHull[ a, b]]
	]


MergeHull[ hl_, hr_] :=
	Block[{hln, hrn},
		hln = HullExtreme[ hl, Max[hl]] ;
		hrn = HullExtreme[ hr, Min[hr]] ;
		{lowl, lowr} = JoinHullFun[ hln, hrn] ; 
		{uppr, uppl} = JoinHullFun[ hrn, hln] ;
		lowl = First[ lowl] ;
		uppr = First[ uppr] ;
		hln = Fold[ If[ Last[#1] === lowl, #1, Append[ #1, #2]]&,
				{First[ uppl]}, Rest[ uppl]] ;
		hrn = Fold[ If[ Last[#1] === uppr, #1, Append[ #1, #2]]&,
				{First[ lowr]}, Rest[ lowr]] ;
		Join[ hln, hrn]
		]


JoinHullFun[ h1_, h2_] :=
	If[ RightOfLine[ Last[h1], First[h1], First[h2]],
			JoinHullFun[ RotateRight[ h1], h2],
			If[ RightOfLine[ First[ h2], Part[ h2, 2], First[ h1]],
				JoinHullFun[ h1, RotateLeft[ h2]],
				{h1, h2}]]


SimpleHull[ a_, b_] :=
	Block[{h, len},
		len = b-a+1;
		h =
		  If[ RightOfLine[ a, a+1, a+2], 
		  			{a, a+2, a+1}, {a,a+1,a+2}] ;
		If[ len > 3, h = AddPoint[ h, a+3]] ;
		If[ len > 4, h = AddPoint[ h, a+4]] ;
		h
		]


AddPoint[ h_, a_] :=
    Block[{hn, hu, hl},
		hn = HullExtreme[ h, Max[ h]] ;
		hu = HullUpp[ hn, a] ;
		hl = First[ HullLow[ hn, a]] ;
		hn = Fold[ If[ Last[#1] === hl, #1, Append[ #1, #2]]&,
				{First[ hu]}, Rest[ hu]] ;
		Append[ hn, a]
		]


HullLow[ h_, a_] :=
	If[ RightOfLine[ First[h], Last[h], a],
				h, HullLow[ RotateRight[ h], a]]
				
HullUpp[ h_, a_] :=
	If[ RightOfLine[ First[h], Part[h, 2], a],
				HullUpp[ RotateLeft[ h], a], h]
				
	
HullExtreme[ h_, hm_] :=
	If[ First[ h] === hm, 
		h, 
		HullExtreme[ RotateLeft[ h], hm]]


RightOfLine[ a_, b_, c_] :=
    Block[{x, y, ac, bc, cc, n},
    	ac = Coord[ a] ;
    	bc = Coord[ b] ;
    	cc = Coord[ c] ;
	 n = bc-ac /. {x_, y_} -> {y, -x} ;
	(cc - ac).n  > 0
	] 


Coord[ a_]  := Part[ points, a]

XCoord[ a_] := Part[ points, a, 1]

YCoord[ a_] := Part[ points, a, 2]



End[]

EndPackage[]		

		
(*:Examples"
	
pts = {{1, 5}, {2,4}, {3,6}, {4,9}, {5,2}, {6,4}}
	
ConvexHull[ pts]
	
HullPlot[ pts]
	
pts = Table[ {Random[], Random[]}, {20}]

ConvexHull[ pts]

p = HullPlot[ pts]

Show[ 
  p /. Point[x_] -> {},
  Graphics[ Table[ Text[ i, Part[ pts, i]], {i,Length[ pts]}]
  ]]
	
*)
			
