#include "y2018/control_loops/python/arm_bounds.h"

#include <CGAL/Exact_predicates_inexact_constructions_kernel.h>
#include <CGAL/Polygon_2_algorithms.h>
#include <CGAL/squared_distance_2.h>
#include <math.h>
#include <iostream>

namespace y2018 {
namespace control_loops {

static BoundsCheck MakeArbitraryArmSpace(::std::vector<Point> points) {
  for (Point &point : points) {
    point = Point(M_PI / 2.0 - point.x(), M_PI / 2.0 - point.y());
  }
  return BoundsCheck(points);
}

Vector GetNormal(Point pt, Segment segment) {
  auto perp_line = segment.supporting_line().perpendicular(pt);
  if (CGAL::do_intersect(perp_line, segment)) {
    return Vector(segment.supporting_line().projection(pt), pt);
  } else {
    if (CGAL::squared_distance(segment.vertex(0), pt) <
        CGAL::squared_distance(segment.vertex(1), pt)) {
      return Vector(segment.vertex(0), pt);
    } else {
      return Vector(segment.vertex(1), pt);
    }
  }
}

double BoundsCheck::min_distance(Point pt,
                                 ::Eigen::Matrix<double, 2, 1> *norm) const {
  auto *cell = grid_.GetCell(pt);
  const auto &points = grid_.points();
  bool inside = false;
  double dist = 0;
  Segment best_segment;
  if (cell) {
    if (cell->IsBorderline()) {
      inside = check_inside(pt, points);
    } else {
      inside = cell->IsInside(pt);
    }
    dist = cell->Distance(pt, &best_segment);
  } else {
    dist = min_dist(pt, points, &best_segment);
    // TODO(austin): Return norm.
  }
  Vector normal = GetNormal(pt, best_segment);
  norm->x() = normal.x();
  norm->y() = normal.y();

  norm->normalize();

  if (!inside) {
    *norm = -(*norm);
  }

  return inside ? -dist : dist;
}

BoundsCheck MakeFullArmSpace() {
  return MakeArbitraryArmSpace({{1.8577014383575772, -1.7353804562372057},
                                {1.8322288438826315, -1.761574570216062},
                                {1.7840339881582052, -1.8122752826851365},
                                {1.7367354460218392, -1.863184376466228},
                                {1.69054633425053, -1.9140647421906793},
                                {1.6456590387358871, -1.9646939485526782},
                                {1.6022412524081968, -2.0148691793459164},
                                {1.5604334435784202, -2.0644108218872432},
                                {1.5203477477575325, -2.1131646494909866},
                                {1.4820681545944325, -2.1610026706370666},
                                {1.4456517763321752, -2.207822815007094},
                                {1.4111309389855604, -2.253547685496041},
                                {1.3785158286698027, -2.298122627340264},
                                {1.3477974448369014, -2.3415133576238922},
                                {1.318950649522341, -2.3837033700108243},
                                {1.2919371475531436, -2.4246912899477153},
                                {1.266708279449626, -2.464488312575651},
                                {1.2432075513427807, -2.5031158147261996},
                                {1.2213728618107609, -2.5406031969824228},
                                {1.2011384131259828, -2.5769859833310096},
                                {1.1824363142639513, -2.61230418457078},
                                {1.165197896140849, -2.64660091671642},
                                {1.149354767212547, -2.6799212560849437},
                                {1.134839641107821, -2.712311307404836},
                                {1.121586968578607, -2.7438174590386177},
                                {1.1095334047223224, -2.774485799309971},
                                {1.09861813993779, -2.8043616692157007},
                                {1.088783119985477, -2.8334893289039758},
                                {1.079973177230731, -2.861911717797479},
                                {1.0721360919151535, -2.8896702908486835},
                                {1.065222599283597, -2.916804915950962},
                                {1.0591863556764607, -2.943353819884641},
                                {1.0539838743128325, -2.969353572295195},
                                {1.049574439440948, -2.9948390990606946},
                                {1.0459200058000553, -3.01984371800932},
                                {1.0429850888932353, -3.044399191310831},
                                {1.0407366503803421, -3.0685357900111128},
                                {1.039143981929867, -3.092282367132034},
                                {1.0381785900853944, -3.1156664365461078},
                                {1.0378140840766794, -3.1387142554815886},
                                {1.038026068010855, -3.1614509090414518},
                                {1.0387920384931424, -3.1839003955494407},
                                {1.0400912884293725, -3.2060857118856374},
                                {1.0419048175385754, -3.2280289382581637},
                                {1.0442152499395827, -3.2497513220895704},
                                {1.047006759060362, -3.2712733608874625},
                                {1.050265000044113, -3.2926148841283163},
                                {1.0539770497854413, -3.3137951343195065},
                                {1.0581313547182551, -3.3348328475243303},
                                {1.0627176864909802, -3.355746333744654},
                                {1.0677271057021644, -3.376553557661551},
                                {1.0731519339297724, -3.397272220341514},
                                {1.0789857343709142, -3.4179198426302158},
                                {1.08522330151719, -3.438513851083329},
                                {1.0918606604276442, -3.4590716674313335},
                                {1.0988950763314502, -3.4796108027504618},
                                {1.1063250755031675, -3.5001489577245133},
                                {1.114150478614587, -3.5207041306442908},
                                {1.1223724480923383, -3.541294735118319},
                                {1.1309935514178655, -3.561939729880535},
                                {1.1400178428208325, -3.5826587636046994},
                                {1.1494509664724388, -3.6034723383075913},
                                {1.1593002851278627, -3.6244019957930833},
                                {1.1695750392617108, -3.6454705327253025},
                                {1.1802865431772855, -3.6667022514168814},
                                {1.191448426478074, -3.6881232554134113},
                                {1.203076931852669, -3.709761801642497},
                                {1.2151912836117884, -3.731648724559698},
                                {1.2278141462274539, -3.753817952785642},
                                {1.2409721988621754, -3.7763071458245956},
                                {1.2546968614661016, -3.7991584885622527},
                                {1.2690252219158191, -3.82241969589366},
                                {1.2840012342034617, -3.846145301494764},
                                {1.2996772887032604, -3.870398337481948},
                                {1.316116303559539, -3.895252562382127},
                                {1.3333945626553043, -3.920795475499501},
                                {1.3516056511178856, -3.9471324882568375},
                                {1.3708660530144583, -3.974392848721103},
                                {1.3913233553973305, -4.002738316267376},
                                {1.4131687110627962, -4.032376331240737},
                                {1.436656614785, -4.063580905628138},
                                {1.4621380338543308, -4.0967276147972065},
                                {1.4901198983671067, -4.132356427437651},
                                {1.5213822452925796, -4.171295431810658},
                                {1.5572408030205178, -4.214938203914619},
                                {1.6002650085871786, -4.266002359325245},
                                {1.657096323745671, -4.331507506063609},
                                {1.7977393629734166, -4.48544594420617},
                                {1.8577014383575772, -4.546720690281509},
                                {1.8577014383575772, -5.497787143782138},
                                {1.2663322229932439, -5.497787143782138},
                                {1.2656247456808565, -5.4967516608259},
                                {1.2287034601894442, -5.445751427768824},
                                {1.1786904071656892, -5.381167803357418},
                                {1.0497616572686415, -5.233423649910944},
                                {0.9097522267255194, -5.096760177110545},
                                {0.8484431816837388, -5.043472544717165},
                                {0.8000049977037023, -5.003989210148714},
                                {0.7581818853313602, -4.9717075285444},
                                {0.7205493452982701, -4.944073122571691},
                                {0.6882849734317291, -4.921474754757885},
                                {0.6882849734317291, -2.40847516476558},
                                {0.8062364039734171, -2.2816832357742984},
                                {0.9846964491122989, -2.0749837238115147},
                                {1.080856081125841, -1.9535526052833936},
                                {1.1403399741223543, -1.8700303125904767},
                                {1.1796460643255697, -1.8073338252603661},
                                {1.206509908068604, -1.7574623871869075},
                                {1.225108933139178, -1.7160975113819317},
                                {1.237917343016644, -1.6806816402603253},
                                {1.2465009152225068, -1.6495957958330445},
                                {1.251901644035212, -1.6217619064422375},
                                {1.2548410275662123, -1.5964327471863136},
                                {1.2558349967266738, -1.5730732293972975},
                                {1.2552624664807475, -1.551289614657788},
                                {1.2534080548485127, -1.5307854126408047},
                                {1.2504896957865235, -1.5113328783804112},
                                {1.2466770509718135, -1.492754008779478},
                                {1.2421041193998992, -1.4749075280685116},
                                {1.236878076935188, -1.457679763958034},
                                {1.231085601347444, -1.4409781183381307},
                                {1.2247974811461413, -1.4247263085140511},
                                {1.2180720288351026, -1.408860841660136},
                                {1.2109576458572935, -1.3933283641188086},
                                {1.2034947755974974, -1.378083641634472},
                                {1.1957174082841977, -1.363088001457586},
                                {1.1876542532510088, -1.3483081171759035},
                                {1.179329661157153, -1.3337150510329991},
                                {1.1707643560768641, -1.3192834919003447},
                                {1.1385726725405734, -1.3512941162901886},
                                {1.1061744838535637, -1.3828092440478137},
                                {1.0736355415504857, -1.4137655512448706},
                                {1.0410203155384732, -1.444102884084807},
                                {1.0083912519665894, -1.4737649313813326},
                                {0.975808099297045, -1.5026998012630925},
                                {0.9433273192311136, -1.5308604887780404},
                                {0.941597772428203, -1.50959779639341},
                                {0.9392183822389457, -1.488861961175901},
                                {0.9362399962318921, -1.4685999567644576},
                                {0.9327074201772598, -1.448764371832554},
                                {0.9286602163651203, -1.4293126388801052},
                                {0.9241333769591611, -1.410206381002334},
                                {0.9191578939466147, -1.3914108561164176},
                                {0.9137612431353617, -1.3728944819981919},
                                {0.9079677963791273, -1.3546284285619337},
                                {0.9017991735984072, -1.3365862662768213},
                                {0.8952745440670551, -1.3187436615861219},
                                {0.888410884742945, -1.3010781117818295},
                                {0.8812232020507231, -1.2835687130679965},
                                {0.8737247224089426, -1.2661959565824437},
                                {0.8659270558807739, -1.2489415479874506},
                                {0.8578403365760008, -1.2317882469234374},
                                {0.8810761292150717, -1.2118184809610988},
                                {0.9041762360363244, -1.1914283310662528},
                                {0.9271196683064211, -1.1706361926784383},
                                {0.949885054142765, -1.1494613527499984},
                                {0.9724507572661958, -1.1279238910069835},
                                {0.9947950004998649, -1.1060445714951375},
                                {1.0168959923240788, -1.0838447258650978},
                                {1.0387320546908576, -1.0613461300367468},
                                {1.0602817502430675, -1.0385708760262022},
                                {1.0815240070740941, -1.0155412408147642},
                                {1.102438239206001, -0.9922795541836177},
                                {1.123004461055328, -0.9688080674301042},
                                {1.1432033942926907, -0.9451488248218312},
                                {1.1630165656794818, -0.9213235395370803},
                                {1.1824263946752058, -0.8973534756890716},
                                {1.2014162698440272, -0.8732593378443982},
                                {1.2199706133398625, -0.8490611692304734},
                                {1.2380749330068292, -0.8247782595916947},
                                {1.2557158618869284, -0.800429063408306},
                                {1.2728811851714203, -0.776031128944234},
                                {1.2895598548592355, -0.7516010383486005},
                                {1.3057419925890068, -0.7271543588072293},
                                {1.3214188812865908, -0.702705604531139},
                                {1.3365829464142562, -0.6782682091832445},
                                {1.351227727719687, -0.6538545081853449},
                                {1.365347842462401, -0.6294757302167044},
                                {1.3789389411433586, -0.6051419971134862},
                                {1.391997656782351, -0.5808623313043475},
                                {1.4045215487801634, -0.5566446698699925},
                                {1.4165090423718034, -0.5324958842910054},
                                {1.4279593646268474, -0.5084218049460658},
                                {1.4388724778869602, -0.4844272494383845},
                                {1.449249011452494, -0.460516053858597},
                                {1.4590901922431447, -0.4366911061340692},
                                {1.468397775065033, -0.4129543806643518},
                                {1.4771739730209745, -0.38930697349737264},
                                {1.485421388504271, -0.36574913735813025},
                                {1.4931429451209484, -0.3422803158986589},
                                {1.5003418207921726, -0.3188991765927855},
                                {1.5070213821985838, -0.2956036417497373},
                                {1.513185120641734, -0.2723909171654731},
                                {1.5188365893149296, -0.24925751796822435},
                                {1.5239793418959948, -0.22619929124396096},
                                {1.5286168722972349, -0.2032114350471563},
                                {1.5327525553319497, -0.1802885134112242},
                                {1.5363895879810432, -0.15742446697009113},
                                {1.5395309308654115, -0.1346126187862511},
                                {1.5421792494481048, -0.11184567494963411},
                                {1.5443368544016174, -0.0891157194637005},
                                {1.5460056404769755, -0.06641420286771664},
                                {1.5471870230984175, -0.043731923953761714},
                                {1.547881871775246, -0.021059003819238205},
                                {1.5480904392645911, 0.0016151486553661097},
                                {1.547812285227133, 0.024301881005710235},
                                {1.5470461928818935, 0.047013352405288866},
                                {1.5457900768723736, 0.06976260156314103},
                                {1.5440408801865422, 0.09256362934244797},
                                {1.5417944575035705, 0.11543149864415554},
                                {1.5390454417383017, 0.13838245474060726},
                                {1.5357870897759938, 0.16143407007284788},
                                {1.5320111023738603, 0.18460541860588778},
                                {1.5277074118667517, 0.20791728625864334},
                                {1.5228639295308124, 0.23139242581719505},
                                {1.5174662420569858, 0.25505586728497265},
                                {1.5114972433117844, 0.27893529808946027},
                                {1.5049366830391921, 0.30306153234932315},
                                {1.4977606078174102, 0.32746909510770755},
                                {1.4899406605544634, 0.35219695697813347},
                                {1.4814431917184283, 0.37728946847450484},
                                {1.4722281161523716, 0.40279756372788583},
                                {1.4622474200998372, 0.4287803341537376},
                                {1.4514431778273647, 0.45530712040457033},
                                {1.4397448652396179, 0.48246034696312606},
                                {1.427065639662639, 0.5103394485717625},
                                {1.4132970536897833, 0.5390664502570618},
                                {1.3983013135749314, 0.5687941401756967},
                                {1.3818995257862978, 0.5997184788509978},
                                {1.3638530549860057, 0.6320982830132342},
                                {1.3438323057823602, 0.6662881915757862},
                                {1.3213606855701236, 0.7027978462604986},
                                {1.2957042956404132, 0.7424084023365868},
                                {1.2656247456808565, 0.786433646353686},
                                {1.2287034601894442, 0.8374338794107618},
                                {1.1786904071656892, 0.902017503822168},
                                {1.0497616572686415, 1.0497616572686426},
                                {0.9097522267255194, 1.1864251300690412},
                                {0.8484431816837388, 1.2397127624624213},
                                {0.8000049977037023, 1.2791960970308718},
                                {0.7581818853313602, 1.3114777786351866},
                                {0.7205493452982701, 1.3391121846078957},
                                {0.6882849734317291, 1.3617105524217008},
                                {0.6882849734317291, 2.356194490192345},
                                {1.3376784164442164, 2.356194490192345},
                                {1.3516056511178856, 2.3360528189227487},
                                {1.3708660530144583, 2.308792458458483},
                                {1.3913233553973305, 2.2804469909122105},
                                {1.4131687110627962, 2.2508089759388485},
                                {1.436656614785, 2.219604401551449},
                                {1.4621380338543308, 2.18645769238238},
                                {1.4901198983671067, 2.1508288797419346},
                                {1.5213822452925796, 2.111889875368928},
                                {1.5572408030205178, 2.0682471032649676},
                                {1.6002650085871786, 2.0171829478543404},
                                {1.657096323745671, 1.9516778011159774},
                                {1.7977393629734166, 1.7977393629734166},
                                {1.8577014383575772, 1.7364646168980775},
                                {1.8577014383575772, -1.7353804562372057}});
}

BoundsCheck MakeClippedArmSpace() {
  return MakeArbitraryArmSpace({{1.8577014383575772, -1.7353804562372057},
                                {1.8322288438826315, -1.761574570216062},
                                {1.7840339881582052, -1.8122752826851365},
                                {1.7367354460218392, -1.863184376466228},
                                {1.69054633425053, -1.9140647421906793},
                                {1.6456590387358871, -1.9646939485526782},
                                {1.6022412524081968, -2.0148691793459164},
                                {1.5604334435784202, -2.0644108218872432},
                                {1.5203477477575325, -2.1131646494909866},
                                {1.4820681545944325, -2.1610026706370666},
                                {1.4456517763321752, -2.207822815007094},
                                {1.4111309389855604, -2.253547685496041},
                                {1.3785158286698027, -2.298122627340264},
                                {1.3477974448369014, -2.3415133576238922},
                                {1.318950649522341, -2.3837033700108243},
                                {1.2919371475531436, -2.4246912899477153},
                                {1.266708279449626, -2.464488312575651},
                                {1.2432075513427807, -2.5031158147261996},
                                {1.2213728618107609, -2.5406031969824228},
                                {1.2011384131259828, -2.5769859833310096},
                                {1.1824363142639513, -2.61230418457078},
                                {1.165197896140849, -2.64660091671642},
                                {1.149354767212547, -2.6799212560849437},
                                {1.134839641107821, -2.712311307404836},
                                {1.121586968578607, -2.7438174590386177},
                                {1.1095334047223224, -2.774485799309971},
                                {1.09861813993779, -2.8043616692157007},
                                {1.088783119985477, -2.8334893289039758},
                                {1.079973177230731, -2.861911717797479},
                                {1.0721360919151535, -2.8896702908486835},
                                {1.065222599283597, -2.916804915950962},
                                {1.0591863556764607, -2.943353819884641},
                                {1.0539838743128325, -2.969353572295195},
                                {1.049574439440948, -2.9948390990606946},
                                {1.0459200058000553, -3.01984371800932},
                                {1.0429850888932353, -3.044399191310831},
                                {1.0407366503803421, -3.0685357900111128},
                                {1.039143981929867, -3.092282367132034},
                                {1.0381785900853944, -3.1156664365461078},
                                {1.0378140840766794, -3.1387142554815886},
                                {1.038026068010855, -3.1614509090414518},
                                {1.0387920384931424, -3.1839003955494407},
                                {1.0400912884293725, -3.2060857118856374},
                                {1.0419048175385754, -3.2280289382581637},
                                {1.0442152499395827, -3.2497513220895704},
                                {1.047006759060362, -3.2712733608874625},
                                {1.050265000044113, -3.2926148841283163},
                                {1.0513266200838918, -3.2986722862692828},
                                {0.6882849734317291, -3.2986722862692828},
                                {0.6882849734317291, -2.40847516476558},
                                {0.8062364039734171, -2.2816832357742984},
                                {0.9846964491122989, -2.0749837238115147},
                                {1.080856081125841, -1.9535526052833936},
                                {1.1403399741223543, -1.8700303125904767},
                                {1.1796460643255697, -1.8073338252603661},
                                {1.206509908068604, -1.7574623871869075},
                                {1.225108933139178, -1.7160975113819317},
                                {1.237917343016644, -1.6806816402603253},
                                {1.2465009152225068, -1.6495957958330445},
                                {1.251901644035212, -1.6217619064422375},
                                {1.2548410275662123, -1.5964327471863136},
                                {1.2558349967266738, -1.5730732293972975},
                                {1.2552624664807475, -1.551289614657788},
                                {1.2534080548485127, -1.5307854126408047},
                                {1.2504896957865235, -1.5113328783804112},
                                {1.2466770509718135, -1.492754008779478},
                                {1.2421041193998992, -1.4749075280685116},
                                {1.236878076935188, -1.457679763958034},
                                {1.231085601347444, -1.4409781183381307},
                                {1.2247974811461413, -1.4247263085140511},
                                {1.2180720288351026, -1.408860841660136},
                                {1.2109576458572935, -1.3933283641188086},
                                {1.2034947755974974, -1.378083641634472},
                                {1.1957174082841977, -1.363088001457586},
                                {1.1876542532510088, -1.3483081171759035},
                                {1.179329661157153, -1.3337150510329991},
                                {1.1707643560768641, -1.3192834919003447},
                                {1.1385726725405734, -1.3512941162901886},
                                {1.1061744838535637, -1.3828092440478137},
                                {1.0736355415504857, -1.4137655512448706},
                                {1.0410203155384732, -1.444102884084807},
                                {1.0083912519665894, -1.4737649313813326},
                                {0.975808099297045, -1.5026998012630925},
                                {0.9433273192311136, -1.5308604887780404},
                                {0.941597772428203, -1.50959779639341},
                                {0.9392183822389457, -1.488861961175901},
                                {0.9362399962318921, -1.4685999567644576},
                                {0.9327074201772598, -1.448764371832554},
                                {0.9286602163651203, -1.4293126388801052},
                                {0.9241333769591611, -1.410206381002334},
                                {0.9191578939466147, -1.3914108561164176},
                                {0.9137612431353617, -1.3728944819981919},
                                {0.9079677963791273, -1.3546284285619337},
                                {0.9017991735984072, -1.3365862662768213},
                                {0.8952745440670551, -1.3187436615861219},
                                {0.888410884742945, -1.3010781117818295},
                                {0.8812232020507231, -1.2835687130679965},
                                {0.8737247224089426, -1.2661959565824437},
                                {0.8659270558807739, -1.2489415479874506},
                                {0.8578403365760008, -1.2317882469234374},
                                {0.8810761292150717, -1.2118184809610988},
                                {0.9041762360363244, -1.1914283310662528},
                                {0.9271196683064211, -1.1706361926784383},
                                {0.949885054142765, -1.1494613527499984},
                                {0.9724507572661958, -1.1279238910069835},
                                {0.9947950004998649, -1.1060445714951375},
                                {1.0168959923240788, -1.0838447258650978},
                                {1.0387320546908576, -1.0613461300367468},
                                {1.0602817502430675, -1.0385708760262022},
                                {1.0815240070740941, -1.0155412408147642},
                                {1.102438239206001, -0.9922795541836177},
                                {1.123004461055328, -0.9688080674301042},
                                {1.1432033942926907, -0.9451488248218312},
                                {1.1630165656794818, -0.9213235395370803},
                                {1.1824263946752058, -0.8973534756890716},
                                {1.2014162698440272, -0.8732593378443982},
                                {1.2199706133398625, -0.8490611692304734},
                                {1.2380749330068292, -0.8247782595916947},
                                {1.2557158618869284, -0.800429063408306},
                                {1.2728811851714203, -0.776031128944234},
                                {1.2895598548592355, -0.7516010383486005},
                                {1.3057419925890068, -0.7271543588072293},
                                {1.3214188812865908, -0.702705604531139},
                                {1.3365829464142562, -0.6782682091832445},
                                {1.351227727719687, -0.6538545081853449},
                                {1.365347842462401, -0.6294757302167044},
                                {1.3789389411433586, -0.6051419971134862},
                                {1.391997656782351, -0.5808623313043475},
                                {1.4045215487801634, -0.5566446698699925},
                                {1.4165090423718034, -0.5324958842910054},
                                {1.4279593646268474, -0.5084218049460658},
                                {1.4388724778869602, -0.4844272494383845},
                                {1.449249011452494, -0.460516053858597},
                                {1.4590901922431447, -0.4366911061340692},
                                {1.468397775065033, -0.4129543806643518},
                                {1.4771739730209745, -0.38930697349737264},
                                {1.485421388504271, -0.36574913735813025},
                                {1.4931429451209484, -0.3422803158986589},
                                {1.5003418207921726, -0.3188991765927855},
                                {1.5070213821985838, -0.2956036417497373},
                                {1.513185120641734, -0.2723909171654731},
                                {1.5188365893149296, -0.24925751796822435},
                                {1.5239793418959948, -0.22619929124396096},
                                {1.5286168722972349, -0.2032114350471563},
                                {1.5327525553319497, -0.1802885134112242},
                                {1.5363895879810432, -0.15742446697009113},
                                {1.5395309308654115, -0.1346126187862511},
                                {1.5421792494481048, -0.11184567494963411},
                                {1.5443368544016174, -0.0891157194637005},
                                {1.5460056404769755, -0.06641420286771664},
                                {1.5471870230984175, -0.043731923953761714},
                                {1.547881871775246, -0.021059003819238205},
                                {1.5480904392645911, 0.0016151486553661097},
                                {1.547812285227133, 0.024301881005710235},
                                {1.5470461928818935, 0.047013352405288866},
                                {1.5457900768723736, 0.06976260156314103},
                                {1.5440408801865422, 0.09256362934244797},
                                {1.5417944575035705, 0.11543149864415554},
                                {1.5390454417383017, 0.13838245474060726},
                                {1.5357870897759938, 0.16143407007284788},
                                {1.5320111023738603, 0.18460541860588778},
                                {1.5277074118667517, 0.20791728625864334},
                                {1.5228639295308124, 0.23139242581719505},
                                {1.5174662420569858, 0.25505586728497265},
                                {1.5114972433117844, 0.27893529808946027},
                                {1.5049366830391921, 0.30306153234932315},
                                {1.4977606078174102, 0.32746909510770755},
                                {1.4899406605544634, 0.35219695697813347},
                                {1.4814431917184283, 0.37728946847450484},
                                {1.4722281161523716, 0.40279756372788583},
                                {1.4622474200998372, 0.4287803341537376},
                                {1.4514431778273647, 0.45530712040457033},
                                {1.4397448652396179, 0.48246034696312606},
                                {1.427065639662639, 0.5103394485717625},
                                {1.4132970536897833, 0.5390664502570618},
                                {1.3983013135749314, 0.5687941401756967},
                                {1.3818995257862978, 0.5997184788509978},
                                {1.3638530549860057, 0.6320982830132342},
                                {1.3438323057823602, 0.6662881915757862},
                                {1.3213606855701236, 0.7027978462604986},
                                {1.2957042956404132, 0.7424084023365868},
                                {1.2656247456808565, 0.786433646353686},
                                {1.2287034601894442, 0.8374338794107618},
                                {1.1786904071656892, 0.902017503822168},
                                {1.0497616572686415, 1.0497616572686426},
                                {0.9097522267255194, 1.1864251300690412},
                                {0.8484431816837388, 1.2397127624624213},
                                {0.8000049977037023, 1.2791960970308718},
                                {0.7581818853313602, 1.3114777786351866},
                                {0.7205493452982701, 1.3391121846078957},
                                {0.6882849734317291, 1.3617105524217008},
                                {0.6882849734317291, 2.356194490192345},
                                {1.3376784164442164, 2.356194490192345},
                                {1.3516056511178856, 2.3360528189227487},
                                {1.3708660530144583, 2.308792458458483},
                                {1.3913233553973305, 2.2804469909122105},
                                {1.4131687110627962, 2.2508089759388485},
                                {1.436656614785, 2.219604401551449},
                                {1.4621380338543308, 2.18645769238238},
                                {1.4901198983671067, 2.1508288797419346},
                                {1.5213822452925796, 2.111889875368928},
                                {1.5572408030205178, 2.0682471032649676},
                                {1.6002650085871786, 2.0171829478543404},
                                {1.657096323745671, 1.9516778011159774},
                                {1.7977393629734166, 1.7977393629734166},
                                {1.8577014383575772, 1.7364646168980775},
                                {1.8577014383575772, -1.7353804562372057}});
}

}  // namespace control_loops
}  // namespace y2018
