/*----------------------------------------------------------------------------*/
/* Copyright (c) 2018-2019 FIRST. All Rights Reserved.                        */
/* Open Source Software - may be modified and shared by FRC teams. The code   */
/* must be accompanied by the FIRST BSD license file in the root directory of */
/* the project.                                                               */
/*----------------------------------------------------------------------------*/

package edu.wpi.cscore.raw;

import edu.wpi.cscore.CameraServerJNI;
import edu.wpi.cscore.ImageSink;

/**
 * A sink for user code to accept video frames as raw bytes.
 *
 * <p>This is a complex API, most cases should use CvSink.
 */
public class RawSink extends ImageSink {
  /**
   * Create a sink for accepting raw images.
   *
   * <p>grabFrame() must be called on the created sink to get each new
   * image.
   *
   * @param name Source name (arbitrary unique identifier)
   */
  public RawSink(String name) {
    super(CameraServerJNI.createRawSink(name));
  }

  /**
   * Wait for the next frame and get the image.
   * Times out (returning 0) after 0.225 seconds.
   * The provided image will have three 8-bit channels stored in BGR order.
   *
   * @return Frame time, or 0 on error (call getError() to obtain the error
   *         message); the frame time is in the same time base as wpi::Now(),
   *         and is in 1 us increments.
   */
  protected long grabFrame(RawFrame frame) {
    return grabFrame(frame, 0.225);
  }

  /**
   * Wait for the next frame and get the image.
   * Times out (returning 0) after timeout seconds.
   * The provided image will have three 8-bit channels stored in BGR order.
   *
   * @return Frame time, or 0 on error (call getError() to obtain the error
   *         message); the frame time is in the same time base as wpi::Now(),
   *         and is in 1 us increments.
   */
  protected long grabFrame(RawFrame frame, double timeout) {
    return CameraServerJNI.grabSinkFrameTimeout(m_handle, frame, timeout);
  }

  /**
   * Wait for the next frame and get the image.  May block forever.
   * The provided image will have three 8-bit channels stored in BGR order.
   *
   * @return Frame time, or 0 on error (call getError() to obtain the error
   *         message); the frame time is in the same time base as wpi::Now(),
   *         and is in 1 us increments.
   */
  protected long grabFrameNoTimeout(RawFrame frame) {
    return CameraServerJNI.grabSinkFrame(m_handle, frame);
  }
}
