#ifndef AOS_EVENTS_LOGGING_LOGFILE_SORTING_H_
#define AOS_EVENTS_LOGGING_LOGFILE_SORTING_H_

#include <iostream>
#include <map>
#include <string>
#include <unordered_map>
#include <unordered_set>
#include <utility>
#include <vector>

#include "aos/configuration.h"
#include "aos/events/logging/file_operations.h"
#include "aos/events/logging/log_backend.h"
#include "aos/time/time.h"
#include "aos/uuid.h"

namespace aos::logger {

struct Boots {
  // Maps the boot UUID to the boot count.  Since boot UUIDs are unique, we
  // don't need to be node specific and can do this for all nodes.
  std::map<std::string, int> boot_count_map;

  // Maps the node index to a set of all boots for that node.
  std::vector<std::vector<std::string>> boots;

  // TODO(austin): Aggregated start time should live here.  This is a property
  // of sorting!
};

// Datastructure to hold ordered parts.
struct LogParts {
  // Monotonic and realtime start times for this set of log files.  For log
  // files which started out unknown and then became known, this is the known
  // start time.
  aos::monotonic_clock::time_point monotonic_start_time;
  aos::realtime_clock::time_point realtime_start_time;

  // Time on the logger node (if applicable) that this log file started.
  aos::monotonic_clock::time_point logger_monotonic_start_time =
      aos::monotonic_clock::min_time;
  aos::realtime_clock::time_point logger_realtime_start_time =
      aos::realtime_clock::min_time;

  // All log files and parts from a single logging event will have
  // the same uuid.  This should be all the files generated on a single node.
  // Used to correlate files recorded together.
  std::string log_event_uuid;
  // All the parts which go together have the same uuid.
  std::string parts_uuid;
  // All log parts generated by a single Logger instance will have the same
  // value here.
  std::string logger_instance_uuid;
  // All log events across all nodes produced by a single high-level start event
  // will have the same value here.
  std::string log_start_uuid;

  // The node this represents, or empty if we are in a single node world.
  std::string node;

  // Boot UUID of the node which generated this data, if available.  For local
  // data and timestamps, this is the same as the logger_boot_uuid.  For remote
  // data, this is the boot_uuid of the remote node.
  std::string source_boot_uuid;

  // Boot number for this node.  This communicates the order of all the
  // source_boot_uuid's for a node.
  size_t boot_count = 0;

  // Boot number for the node where this data was logged.
  // This is theoretically redundant with LogFile, except that we quickly end up
  // using LogParts without the corresponding LogFile datastructure.
  size_t logger_boot_count = 0;

  // Pre-sorted list of parts.
  std::vector<std::string> parts;

  // Configuration for all the log parts.  This will be a single object for all
  // log files with the same config.
  std::string config_sha256;
  std::shared_ptr<const aos::Configuration> config;

  // Information about all the boots that the system has observed.
  std::shared_ptr<const Boots> boots;

  // Highest max out of order durations among all parts.
  std::chrono::nanoseconds max_out_of_order_duration;

  // The types of data we know are in this log file.  If we have no info, this
  // will be a vector of all the potential types.
  std::vector<StoredDataType> data_stored;
};

// Datastructure to hold parts from the same run of the logger which have no
// ordering constraints relative to each other.
struct LogFile {
  // All log files and parts from a single logging event will have
  // the same uuid.  This should be all the files generated on a single node.
  // Used to correlate files recorded together.
  std::string log_event_uuid;
  // All log parts generated by a single Logger instance will have the same
  // value here.
  std::string logger_instance_uuid;
  // All log events across all nodes produced by a single high-level start event
  // will have the same value here.
  std::string log_start_uuid;

  // The node the logger was running on (if available)
  std::string logger_node;
  // Boot UUID of the node running the logger.
  std::string logger_boot_uuid;

  // Boot number for the logger node.
  size_t logger_boot_count = 0;

  // The start time on the logger node.
  aos::monotonic_clock::time_point monotonic_start_time;
  aos::realtime_clock::time_point realtime_start_time;

  // The name field in the log file headers.
  std::string name;

  // The logger version info in the logfile headers, if available.
  std::string logger_sha1;
  std::string logger_version;

  // All the parts, unsorted.
  std::vector<LogParts> parts;

  // A list of parts which were corrupted and are unknown where they should go.
  std::vector<std::string> corrupted;

  // Configuration for all the log parts and files.  This will be a single
  // object for log files with the same config.
  std::string config_sha256;
  std::shared_ptr<const aos::Configuration> config;
  std::shared_ptr<const Boots> boots;
};

std::ostream &operator<<(std::ostream &stream, const LogFile &file);
std::ostream &operator<<(std::ostream &stream, const LogParts &parts);

// Takes a bunch of parts and sorts them based on part_uuid and part_index.
std::vector<LogFile> SortParts(const std::vector<std::string> &parts);
std::vector<LogFile> SortParts(
    const std::vector<internal::FileOperations::File> &parts);

// Sort parts of a single log.
std::vector<LogFile> SortParts(const LogSource &log_source);

// Returns a list of all the logger nodes for the specified set of log files.
// For single-node systems, the empty string will represent the logfile set.
std::set<std::string> LoggerNodes(const std::vector<LogFile> &log_files);

// Validates that collection of log files or log parts shares the same configs.
bool HasMatchingConfigs(const std::vector<LogFile> &items);

// Recursively searches the file/folder for .bfbs and .bfbs.xz files and adds
// them to the vector.
void FindLogs(std::vector<internal::FileOperations::File> *files,
              std::string filename);

// Recursively searches the file/folder for .bfbs and .bfbs.xz files and returns
// them in a vector.
std::vector<internal::FileOperations::File> FindLogs(std::string filename);

// Recursively searches for logfiles in argv[1] and onward.
std::vector<internal::FileOperations::File> FindLogs(int argc, char **argv);

// Proxy container to bind log parts with log source. It helps with reading logs
// from virtual media such as memory or S3.
class LogPartsAccess {
 public:
  LogPartsAccess(std::optional<const LogSource *> log_source,
                 LogParts log_parts)
      : log_source_(std::move(log_source)), log_parts_(std::move(log_parts)) {
    CHECK(!log_parts_.parts.empty());
  }

  std::string_view node_name() const { return log_parts_.node; }

  std::string_view source_boot_uuid() const {
    return log_parts_.source_boot_uuid;
  }

  size_t boot_count() const { return log_parts_.boot_count; }

  std::shared_ptr<const Configuration> config() const {
    return log_parts_.config;
  }

  std::optional<const LogSource *> log_source() const { return log_source_; }

  std::chrono::nanoseconds max_out_of_order_duration() const {
    return log_parts_.max_out_of_order_duration;
  }

  std::string GetPartAt(size_t index) const {
    CHECK_LT(index, log_parts_.parts.size());
    return log_parts_.parts[index];
  }

  // TODO (Alexei): do we need to reduce it to concrete operations?
  const LogParts &parts() const { return log_parts_; }

  size_t size() const { return log_parts_.parts.size(); }

 private:
  std::optional<const LogSource *> log_source_;
  LogParts log_parts_;
};

// Output of LogPartsAccess for debug purposes.
std::ostream &operator<<(std::ostream &stream,
                         const LogPartsAccess &log_parts_access);

// Collection of log parts that associated with pair: node and boot.
class SelectedLogParts {
 public:
  SelectedLogParts(std::string_view node_name, size_t boot_index,
                   std::vector<LogPartsAccess> log_parts);

  // Use items in fancy loops.
  auto begin() const { return log_parts_.begin(); }
  auto end() const { return log_parts_.end(); }

  bool empty() const { return log_parts_.empty(); }

  // Config that shared across all log parts.
  std::shared_ptr<const Configuration> config() const { return config_; }

  const std::string &node_name() const { return node_name_; }

  // Returns the boot index found in the log parts.
  size_t boot_index() const { return boot_index_; }

 private:
  std::string node_name_;
  size_t boot_index_ = 0u;
  std::vector<LogPartsAccess> log_parts_;

  std::shared_ptr<const Configuration> config_ = nullptr;
};

// Container that keeps a sorted list of log files and provides functions that
// commonly used during log reading.
class LogFilesContainer {
 public:
  // Initializes log file container with the list of sorted files (results of
  // SortParts).
  explicit LogFilesContainer(std::vector<LogFile> log_files)
      : LogFilesContainer(std::nullopt, std::move(log_files)) {}

  // Sorts and initializes log container with files from an abstract log source.
  explicit LogFilesContainer(const LogSource *log_source)
      : LogFilesContainer(log_source, SortParts(*log_source)) {}

  // Returns true when at least one of the log files associated with node.
  bool ContainsPartsForNode(std::string_view node_name) const {
    // TODO (Alexei): Implement
    // https://en.cppreference.com/w/cpp/container/unordered_map/find with C++20
    return nodes_boots_.count(std::string(node_name)) > 0;
  }

  // Returns numbers of reboots found in log files associated with the node.
  size_t BootsForNode(std::string_view node_name) const;

  // Get only log parts that associated with node, boot number, and with data of
  // any of the types provided.
  SelectedLogParts SelectParts(std::string_view node_name, size_t boot_index,
                               const std::vector<StoredDataType> &types) const;

  // It provides access to boots logged by all log files in the container.
  const std::shared_ptr<const Boots> &boots() const { return boots_; }

  // Access the configuration shared with all log files in the container.
  std::shared_ptr<const Configuration> config() const { return config_; }

  // List of logger nodes for given set of log files.
  const auto &logger_nodes() const { return logger_nodes_; }

  // TODO (Alexei): it is not clear what it represents for multiple log events.
  // Review its usage.
  std::string_view name() const { return log_files_[0].name; }

  // Returns true if the timestamps (remote and local) are stored only in files
  // distinct from the data.
  bool TimestampsStoredSeparately() const;

  // Returns true if we have timestamps in any of the files.
  bool HasTimestamps(std::string_view node_name) const;

  const std::vector<LogFile> &log_files() const { return log_files_; }

 private:
  LogFilesContainer(std::optional<const LogSource *> log_source,
                    std::vector<LogFile> log_files);

  std::optional<const LogSource *> log_source_;
  std::vector<LogFile> log_files_;

  std::shared_ptr<const Configuration> config_;
  std::shared_ptr<const Boots> boots_;

  // Keeps information about nodes and number of reboots per node.
  std::unordered_map<std::string, size_t> nodes_boots_;
  std::vector<std::string> logger_nodes_;
};

}  // namespace aos::logger

#endif  // AOS_EVENTS_LOGGING_LOGFILE_SORTING_H_
