/*
    FreeRTOS V6.0.5 - Copyright (C) 2010 Real Time Engineers Ltd.

    ***************************************************************************
    *                                                                         *
    * If you are:                                                             *
    *                                                                         *
    *    + New to FreeRTOS,                                                   *
    *    + Wanting to learn FreeRTOS or multitasking in general quickly       *
    *    + Looking for basic training,                                        *
    *    + Wanting to improve your FreeRTOS skills and productivity           *
    *                                                                         *
    * then take a look at the FreeRTOS eBook                                  *
    *                                                                         *
    *        "Using the FreeRTOS Real Time Kernel - a Practical Guide"        *
    *                  http://www.FreeRTOS.org/Documentation                  *
    *                                                                         *
    * A pdf reference manual is also available.  Both are usually delivered   *
    * to your inbox within 20 minutes to two hours when purchased between 8am *
    * and 8pm GMT (although please allow up to 24 hours in case of            *
    * exceptional circumstances).  Thank you for your support!                *
    *                                                                         *
    ***************************************************************************

    This file is part of the FreeRTOS distribution.

    FreeRTOS is free software; you can redistribute it and/or modify it under
    the terms of the GNU General Public License (version 2) as published by the
    Free Software Foundation AND MODIFIED BY the FreeRTOS exception.
    ***NOTE*** The exception to the GPL is included to allow you to distribute
    a combined work that includes FreeRTOS without being obliged to provide the
    source code for proprietary components outside of the FreeRTOS kernel.
    FreeRTOS is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details. You should have received a copy of the GNU General Public
    License and the FreeRTOS license exception along with FreeRTOS; if not it
    can be viewed here: http://www.freertos.org/a00114.html and also obtained
    by writing to Richard Barry, contact details for whom are available on the
    FreeRTOS WEB site.

    1 tab == 4 spaces!

    http://www.FreeRTOS.org - Documentation, latest information, license and
    contact details.

    http://www.SafeRTOS.com - A version that is certified for use in safety
    critical systems.

    http://www.OpenRTOS.com - Commercial support, development, porting,
    licensing and training services.
*/

/*
 * This is the list implementation used by the scheduler.  While it is tailored
 * heavily for the schedulers needs, it is also available for use by
 * application code.
 *
 * xLists can only store pointers to xListItems.  Each xListItem contains a
 * numeric value (xItemValue).  Most of the time the lists are sorted in
 * descending item value order.
 *
 * Lists are created already containing one list item.  The value of this
 * item is the maximum possible that can be stored, it is therefore always at
 * the end of the list and acts as a marker.  The list member pxHead always
 * points to this marker - even though it is at the tail of the list.  This
 * is because the tail contains a wrap back pointer to the true head of
 * the list.
 *
 * In addition to it's value, each list item contains a pointer to the next
 * item in the list (pxNext), a pointer to the list it is in (pxContainer)
 * and a pointer to back to the object that contains it.  These later two
 * pointers are included for efficiency of list manipulation.  There is
 * effectively a two way link between the object containing the list item and
 * the list item itself.
 *
 *
 * \page ListIntroduction List Implementation
 * \ingroup FreeRTOSIntro
 */

/*
	Changes from V4.3.1

	+ Included local const within listGET_OWNER_OF_NEXT_ENTRY() to assist
	  compiler with optimisation.  Thanks B.R.
*/

#ifndef LIST_H
#define LIST_H

#ifdef __cplusplus
extern "C"
{
#endif
	/*
	 * Definition of the only type of object that a list can contain.
	 */
	struct xLIST_ITEM {
		portTickType xItemValue;				/*< The value being listed.  In most cases this is used to sort the list in descending order. */
		volatile struct xLIST_ITEM * pxNext;	/*< Pointer to the next xListItem in the list. */
		volatile struct xLIST_ITEM * pxPrevious;/*< Pointer to the previous xListItem in the list. */
		void * pvOwner;							/*< Pointer to the object (normally a TCB) that contains the list item.  There is therefore a two way link between the object containing the list item and the list item itself. */
		void * pvContainer;						/*< Pointer to the list in which this list item is placed (if any). */
	};
	typedef struct xLIST_ITEM xListItem;		/* For some reason lint wants this as two separate definitions. */

	struct xMINI_LIST_ITEM {
		portTickType xItemValue;
		volatile struct xLIST_ITEM *pxNext;
		volatile struct xLIST_ITEM *pxPrevious;
	};
	typedef struct xMINI_LIST_ITEM xMiniListItem;

	/*
	 * Definition of the type of queue used by the scheduler.
	 */
	typedef struct xLIST {
		volatile unsigned portBASE_TYPE uxNumberOfItems;
		volatile xListItem * pxIndex;			/*< Used to walk through the list.  Points to the last item returned by a call to pvListGetOwnerOfNextEntry (). */
		volatile xMiniListItem xListEnd;		/*< List item that contains the maximum possible item value meaning it is always at the end of the list and is therefore used as a marker. */
	} xList;

	/*
	 * Access macro to set the owner of a list item.  The owner of a list item
	 * is the object (usually a TCB) that contains the list item.
	 *
	 * \page listSET_LIST_ITEM_OWNER listSET_LIST_ITEM_OWNER
	 * \ingroup LinkedList
	 */
#define listSET_LIST_ITEM_OWNER( pxListItem, pxOwner )		( pxListItem )->pvOwner = ( void * ) pxOwner

	/*
	 * Access macro to set the value of the list item.  In most cases the value is
	 * used to sort the list in descending order.
	 *
	 * \page listSET_LIST_ITEM_VALUE listSET_LIST_ITEM_VALUE
	 * \ingroup LinkedList
	 */
#define listSET_LIST_ITEM_VALUE( pxListItem, xValue )		( pxListItem )->xItemValue = xValue

	/*
	 * Access macro the retrieve the value of the list item.  The value can
	 * represent anything - for example a the priority of a task, or the time at
	 * which a task should be unblocked.
	 *
	 * \page listGET_LIST_ITEM_VALUE listGET_LIST_ITEM_VALUE
	 * \ingroup LinkedList
	 */
#define listGET_LIST_ITEM_VALUE( pxListItem )				( ( pxListItem )->xItemValue )

	/*
	 * Access macro to determine if a list contains any items.  The macro will
	 * only have the value true if the list is empty.
	 *
	 * \page listLIST_IS_EMPTY listLIST_IS_EMPTY
	 * \ingroup LinkedList
	 */
#define listLIST_IS_EMPTY( pxList )				( ( pxList )->uxNumberOfItems == ( unsigned portBASE_TYPE ) 0 )

	/*
	 * Access macro to return the number of items in the list.
	 */
#define listCURRENT_LIST_LENGTH( pxList )		( ( pxList )->uxNumberOfItems )

	/*
	 * Access function to obtain the owner of the next entry in a list.
	 *
	 * The list member pxIndex is used to walk through a list.  Calling
	 * listGET_OWNER_OF_NEXT_ENTRY increments pxIndex to the next item in the list
	 * and returns that entries pxOwner parameter.  Using multiple calls to this
	 * function it is therefore possible to move through every item contained in
	 * a list.
	 *
	 * The pxOwner parameter of a list item is a pointer to the object that owns
	 * the list item.  In the scheduler this is normally a task control block.
	 * The pxOwner parameter effectively creates a two way link between the list
	 * item and its owner.
	 *
	 * @param pxList The list from which the next item owner is to be returned.
	 *
	 * \page listGET_OWNER_OF_NEXT_ENTRY listGET_OWNER_OF_NEXT_ENTRY
	 * \ingroup LinkedList
	 */
#define listGET_OWNER_OF_NEXT_ENTRY( pxTCB, pxList )									\
{																						\
xList * const pxConstList = pxList;														\
	/* Increment the index to the next item and return the item, ensuring */			\
	/* we don't return the marker used at the end of the list.  */						\
	( pxConstList )->pxIndex = ( pxConstList )->pxIndex->pxNext;						\
	if( ( pxConstList )->pxIndex == ( xListItem * ) &( ( pxConstList )->xListEnd ) )	\
	{																					\
		( pxConstList )->pxIndex = ( pxConstList )->pxIndex->pxNext;					\
	}																					\
	pxTCB = ( pxConstList )->pxIndex->pvOwner;											\
}


	/*
	 * Access function to obtain the owner of the first entry in a list.  Lists
	 * are normally sorted in ascending item value order.
	 *
	 * This function returns the pxOwner member of the first item in the list.
	 * The pxOwner parameter of a list item is a pointer to the object that owns
	 * the list item.  In the scheduler this is normally a task control block.
	 * The pxOwner parameter effectively creates a two way link between the list
	 * item and its owner.
	 *
	 * @param pxList The list from which the owner of the head item is to be
	 * returned.
	 *
	 * \page listGET_OWNER_OF_HEAD_ENTRY listGET_OWNER_OF_HEAD_ENTRY
	 * \ingroup LinkedList
	 */
#define listGET_OWNER_OF_HEAD_ENTRY( pxList )  ( ( pxList->uxNumberOfItems != ( unsigned portBASE_TYPE ) 0 ) ? ( (&( pxList->xListEnd ))->pxNext->pvOwner ) : ( NULL ) )

	/*
	 * Check to see if a list item is within a list.  The list item maintains a
	 * "container" pointer that points to the list it is in.  All this macro does
	 * is check to see if the container and the list match.
	 *
	 * @param pxList The list we want to know if the list item is within.
	 * @param pxListItem The list item we want to know if is in the list.
	 * @return pdTRUE is the list item is in the list, otherwise pdFALSE.
	 * pointer against
	 */
#define listIS_CONTAINED_WITHIN( pxList, pxListItem ) ( ( pxListItem )->pvContainer == ( void * ) pxList )

	/*
	 * Must be called before a list is used!  This initialises all the members
	 * of the list structure and inserts the xListEnd item into the list as a
	 * marker to the back of the list.
	 *
	 * @param pxList Pointer to the list being initialised.
	 *
	 * \page vListInitialise vListInitialise
	 * \ingroup LinkedList
	 */
	void vListInitialise(xList *pxList);

	/*
	 * Must be called before a list item is used.  This sets the list container to
	 * null so the item does not think that it is already contained in a list.
	 *
	 * @param pxItem Pointer to the list item being initialised.
	 *
	 * \page vListInitialiseItem vListInitialiseItem
	 * \ingroup LinkedList
	 */
	void vListInitialiseItem(xListItem *pxItem);

	/*
	 * Insert a list item into a list.  The item will be inserted into the list in
	 * a position determined by its item value (descending item value order).
	 *
	 * @param pxList The list into which the item is to be inserted.
	 *
	 * @param pxNewListItem The item to that is to be placed in the list.
	 *
	 * \page vListInsert vListInsert
	 * \ingroup LinkedList
	 */
	void vListInsert(xList *pxList, xListItem *pxNewListItem);

	/*
	 * Insert a list item into a list.  The item will be inserted in a position
	 * such that it will be the last item within the list returned by multiple
	 * calls to listGET_OWNER_OF_NEXT_ENTRY.
	 *
	 * The list member pvIndex is used to walk through a list.  Calling
	 * listGET_OWNER_OF_NEXT_ENTRY increments pvIndex to the next item in the list.
	 * Placing an item in a list using vListInsertEnd effectively places the item
	 * in the list position pointed to by pvIndex.  This means that every other
	 * item within the list will be returned by listGET_OWNER_OF_NEXT_ENTRY before
	 * the pvIndex parameter again points to the item being inserted.
	 *
	 * @param pxList The list into which the item is to be inserted.
	 *
	 * @param pxNewListItem The list item to be inserted into the list.
	 *
	 * \page vListInsertEnd vListInsertEnd
	 * \ingroup LinkedList
	 */
	void vListInsertEnd(xList *pxList, xListItem *pxNewListItem);

	/*
	 * Remove an item from a list.  The list item has a pointer to the list that
	 * it is in, so only the list item need be passed into the function.
	 *
	 * @param vListRemove The item to be removed.  The item will remove itself from
	 * the list pointed to by it's pxContainer parameter.
	 *
	 * \page vListRemove vListRemove
	 * \ingroup LinkedList
	 */
	void vListRemove(xListItem *pxItemToRemove);

#ifdef __cplusplus
}
#endif

#endif

