(* :Name: ConstrainedContour` *)

(* :Title: ConstrainedContour *)

(* :Author: Tom Wickham-Jones*)

(* :Package Version: 1.0 *)

(* :Mathematica Version: 2.2 *)

(*:Summary:
	This package supports the function ConstrainedContourPlot.
	This draws contours subject to a constraint.
*)


(* :History:
	Created summer 1994 by Tom Wickham-Jones.
	
	This package is described in the book
	Mathematica Graphics: Techniques and Applications.
	Tom Wickham-Jones, TELOS/Springer-Verlag 1994.

*)



BeginPackage[ "ExtendGraphics`ConstrainedContour`", 
				"Utilities`FilterOptions`"]

ConstrainedContourPlot::usage =
	"ConstrainedContourPlot[ f, c, x-rng, y-rng] 
plots contours of f over x-rng and y-rng subject to the constraint c."

Constraint::usage =
	"Constraint is an option of ConstrainedContourPlot that
	determines how the constraint is plotted.   A value of
	False or None draws no contraint.  A value of True
	or Automatic draws the constraint.  A setting with a
	style will draw the constraint in a particular style."

Begin[ "`Private`"]


FindContours[ x_List, {z1_, z2_}] := x

FindContours[ n_Integer /; n > 0, {z1_, z2_}] :=
    Block[{zinc},
        zinc = (z2 - z1)/ (n + 1) ;
        Table[ i, {i, z1 + zinc, z2 - zinc, zinc}]
        ]

TestMessage[ sym_, tag_] := 
	If[ Head[ MessageName[sym, tag]] === $Off, False, Off[ MessageName[sym, tag]]; True]

Options[ ConstrainedContourPlot] =
	Join[ Options[ ContourPlot], 
			{Constraint -> Automatic}]

SetOptions[ ConstrainedContourPlot, 
	 AspectRatio -> Automatic,
	 PlotPoints -> 30]
	
ConstrainedContourPlot[f_, con_, {x_, x0_, x1_}, {y_, y0_, y1_}, opts___] :=
    Block[{num, style, cont, prng, prims, 
    		cons, conplot, opt, on1, on2, cstyle, mstyle, nconts},
    	num = Contours /. {opts} /. Options[ ConstrainedContourPlot] ;
    	cstyle = Constraint /. {opts} /. Options[ ConstrainedContourPlot] ;
    	mstyle = ContourStyle /. {opts} /. Options[ ConstrainedContourPlot] ;
		nconts = If[ ListQ[ num], Length[ num], num] ;
		mstyle = FixStyle[ mstyle, nconts] ;
		cstyle = FixConstraint[ cstyle] ;
    	style = Table[ {Thickness[ i]}, {i, nconts}] ;
		opt = FilterOptions[ ContourPlot,
			Apply[ Sequence, Join[{opts}, Options[ ConstrainedContourPlot]]]] ;
    	cont = ContourPlot[ f, {x, x0, x1}, {y, y0, y1},
    				ContourStyle -> style,
					ContourShading -> False,
    				DisplayFunction -> Identity, 
    				Evaluate[ opt]] ;
    	prng = FullOptions[ cont, PlotRange] ;
    	cont = Graphics[ cont] ;
		vals = FindContours[ num, Last[ prng]] ;
    	prims = First[ cont] ;
     	prims = prims /. 
					Thickness[i_] :> {Part[ mstyle, i],Part[ vals, i]} ;
   		copt = Rest[ cont] ;
    	cons = GetConstraint[ N[ con]] ;
		conplot = {} ;
		If[ cstyle =!= None,
	    	conplot = ContourPlot[ cons, {x, x0, x1}, {y,y0,y1},
				Contours -> {0.0},
				ContourShading -> False,
				ContourStyle -> cstyle,
				DisplayFunction -> Identity,
				Evaluate[ opt]]] ;
		on1 = TestMessage[ LinearSolve, "luc"] ;
		on2 = TestMessage[ LinearSolve, "sing"] ;
    	prims = Map[ ApplyConstraint[#[[2,1]], #[[3,1]], f - #[[2,2]], cons, x, y]&, prims] ;
		If[ on1, On[ LinearSolve::luc]] ;
		If[ on2, On[ LinearSolve::sing]] ;
    	Show[ Insert[ copt, prims, 1], 
			conplot,
				PlotRange -> Drop[ prng, -1],
    			DisplayFunction -> $DisplayFunction]
   ]

FixConstraint[ cstyle_] :=
	Switch[ cstyle,
				True, Automatic,
				Automatic, Automatic,
				None, None,
				False, None,
				_, cstyle]

FixStyle[ {s1__}, nconts_] :=
	If[ !ListQ[ First[ {s1}]],
		Table[ {s1}, {i,nconts}],
		Table[ Part[ {s1}, Mod[ i-1, Length[{s1}]]+1], {i,nconts}]]

FixStyle[ Automatic, nconts_] := FixStyle[ {{}}, nconts]

FixStyle[ style_, nconts_] := FixStyle[ {style}, nconts]

GetConstraint[ Greater[left_, right_]] :=
	left-right
	
GetConstraint[ Less[left_, right_]] :=
	right-left
	
GetConstraint[ _] :=
	(
	Message[ ConsContourPlot::badcons];
	1.0
	)
	
ConsContourPlot::badcons = 
	"Constraint is not of the form f[x, y] > a or
	 f[x, y] < b.";
    	
ApplyConstraint[ style_, pts_, f_, c_, x_, y_] :=
    Block[{res, len},
	res = Map[ ((c /. {x -> #[[1]], y -> #[[2]]}) > 0.0)&, pts] ;
	len = Length[ pts] ;
	res =
	    Table[ 
  		If[ 
		   res[[i]], 
		     {pts[[i]]},
		     {
		     If[ 
			(i =!= 1   && res[[i-1]]), 
			    GetRoot[ f, c, x, y, pts[[i]], pts[[i-1]]], False],
		     If[ 
			(i =!= len   && res[[i+1]]), 
			    GetRoot[ f, c, x, y, pts[[i]], pts[[i+1]]], False]
		     }],
		{i,len}] ;
	res = Flatten[ res, 1] ;
	res = 
	   Which[
	       FreeQ[ res, False], {res},
	       Union[res] === {False}, {},
	       True,
		res = res //. {a___, False, False, b___} -> {a, False, b} ;
		res = Fold[ 
			If[ #2 === False, 
		 		Append[#1, {}], 
		 		Insert[#1, #2, {-1,-1}]]&,{{}}, res] ;
		res = DeleteCases[ res, {}]];
	Join[style, Map[ Line, res]]
    ]

GetRoot[ f_, c_, x_, y_, {x0_, y0_}, {x1_, y1_}] :=
    Block[{xs0, xs1, ys0, ys1},
    	If[ x0 === x1,
		xs0 = x0 0.99; xs1 = x1 1.01,
		xs0 = x0; xs1 = x1] ;
    	If[ y0 === y1,
		ys0 = y0 0.99; ys1 = y1 1.01,
		ys0 = y0; ys1 = y1] ;
		{x, y} /. FindRoot[ {f == 0, c == 0}, {x, xs0, xs1}, {y,ys0,ys1}]
		]


End[]

EndPackage[]

(*:Examples:


<<ConstrainedContour.m

ConstrainedContourPlot[ x y, y > x^2 - 1, 
			{x, -1.5, 1.5}, {y,-1.5, 1.5}, 
			 Contours -> {-1.},
			 ContourShading -> False,
				PlotPoints -> 10]


ConstrainedContourPlot[ x y, y > x^2 - 1, 
			{x, -1.5, 1.5}, {y,-1.5, 1.5}, 
			 Contours -> 5,
			 ContourShading -> False,
				PlotPoints -> 10]


ConstrainedContourPlot[ Sin[x^2 + y], x^2 + 3 y^2 < 1.6, 
                        {x, -1.5, 1.5}, {y,-1.5, 1.5}, 
                                PlotPoints -> 30]



*)

