#include "options.h"
#include "../ice_server/server.h"
#include <rawrtc/ice_gather_options.h>
#include <rawrtcc/code.h>
#include <re.h>

/*
 * Destructor for an existing ICE gather options instance.
 */
static void rawrtc_ice_gather_options_destroy(void* arg) {
    struct rawrtc_ice_gather_options* const options = arg;

    // Un-reference
    list_flush(&options->ice_servers);
}

/*
 * Create a new ICE gather options instance.
 * `*optionsp` must be unreferenced.
 */
enum rawrtc_code rawrtc_ice_gather_options_create(
    struct rawrtc_ice_gather_options** const optionsp,  // de-referenced
    enum rawrtc_ice_gather_policy const gather_policy) {
    struct rawrtc_ice_gather_options* options;

    // Check arguments
    if (!optionsp) {
        return RAWRTC_CODE_INVALID_ARGUMENT;
    }

    // Allocate
    options = mem_zalloc(sizeof(*options), rawrtc_ice_gather_options_destroy);
    if (!options) {
        return RAWRTC_CODE_NO_MEMORY;
    }

    // Set fields/reference
    options->gather_policy = gather_policy;
    list_init(&options->ice_servers);
    options->udp_port_range.min = 0;
    options->udp_port_range.max = 0;

    // Set pointer and return
    *optionsp = options;
    return RAWRTC_CODE_SUCCESS;
}

enum rawrtc_code rawrtc_ice_gather_options_set_udp_port_range(
    struct rawrtc_ice_gather_options* const options,
    uint16_t min_udp_port,
    uint16_t max_udp_port) {

    if (!options) {
        return RAWRTC_CODE_INVALID_ARGUMENT;
    }

    if (max_udp_port < min_udp_port) {
        return RAWRTC_CODE_INVALID_ARGUMENT;
    }

    options->udp_port_range.min = min_udp_port;
    options->udp_port_range.max = max_udp_port;

    return RAWRTC_CODE_SUCCESS;
}

/*
 * Add an ICE server instance to the gather options.
 */
enum rawrtc_code rawrtc_ice_gather_options_add_server_internal(
    struct rawrtc_ice_gather_options* const options, struct rawrtc_ice_server* const server) {
    // Check arguments
    if (!options || !server) {
        return RAWRTC_CODE_INVALID_ARGUMENT;
    }

    // Add to options
    list_append(&options->ice_servers, &server->le, server);
    return RAWRTC_CODE_SUCCESS;
}

/*
 * Add an ICE server to the gather options.
 */
enum rawrtc_code rawrtc_ice_gather_options_add_server(
    struct rawrtc_ice_gather_options* const options,
    const char* const* const urls,  // copied
    size_t const n_urls,
    const char* const username,  // nullable, copied
    const char* const credential,  // nullable, copied
    enum rawrtc_ice_credential_type const credential_type) {
    struct rawrtc_ice_server* server;
    enum rawrtc_code error;

    // Check arguments
    if (!options) {
        return RAWRTC_CODE_INVALID_ARGUMENT;
    }

    // Ensure there are less than 2^8 servers
    // TODO: This check should be in some common location
    if (list_count(&options->ice_servers) == UINT8_MAX) {
        return RAWRTC_CODE_INSUFFICIENT_SPACE;
    }

    // Create ICE server
    error = rawrtc_ice_server_create(&server, urls, n_urls, username, credential, credential_type);
    if (error) {
        return error;
    }

    // Add to options
    return rawrtc_ice_gather_options_add_server_internal(options, server);
}
