#include <functional>

#include "aos/common/logging/logging.h"

#include "frc971/actions/shoot_action.h"
#include "frc971/control_loops/shooter/shooter.q.h"
#include "frc971/control_loops/claw/claw.q.h"
#include "frc971/constants.h"
#include "frc971/control_loops/drivetrain/drivetrain.q.h"

namespace frc971 {
namespace actions {
namespace {

bool IntakeOff() {
  control_loops::claw_queue_group.goal.FetchLatest();
  if (!control_loops::claw_queue_group.goal.get()) {
    LOG(WARNING, "no claw goal\n");
    // If it doesn't have a goal, then the intake isn't on so we don't have to
    // turn it off.
    return true;
  } else {
    if (!control_loops::claw_queue_group.goal.MakeWithBuilder()
        .bottom_angle(control_loops::claw_queue_group.goal->bottom_angle)
        .separation_angle(
            control_loops::claw_queue_group.goal->separation_angle)
        .intake(0.0)
        .centering(0.0)
        .Send()) {
      LOG(WARNING, "sending claw goal failed\n");
      return false;
    }
  }
  return true;
}

}  // namespace

constexpr double ShootAction::kOffsetRadians;
constexpr double ShootAction::kClawShootingSeparation;
constexpr double ShootAction::kClawShootingSeparationGoal;

ShootAction::ShootAction(actions::ShootActionQueueGroup* s)
    : actions::ActionBase<actions::ShootActionQueueGroup>(s) {}

double ShootAction::SpeedToAngleOffset(double speed) {
  const frc971::constants::Values& values = frc971::constants::GetValues();
  // scale speed to a [0.0-1.0] on something close to the max
  return (speed / values.drivetrain_max_speed) * ShootAction::kOffsetRadians;
}
void ShootAction::RunAction() {
  InnerRunAction();

  // Now do our 'finally' block and make sure that we aren't requesting shots
  // continually.
  control_loops::shooter_queue_group.goal.FetchLatest();
  if (control_loops::shooter_queue_group.goal.get() == nullptr) {
    return;
  }
  if (!control_loops::shooter_queue_group.goal.MakeWithBuilder()
           .shot_power(control_loops::shooter_queue_group.goal->shot_power)
           .shot_requested(false)
           .unload_requested(false)
           .load_requested(false)
           .Send()) {
    LOG(WARNING, "sending shooter goal failed\n");
    return;
  }

  LOG(INFO, "finished\n");
}

void ShootAction::InnerRunAction() {
  LOG(INFO, "Shooting at the current angle and power.\n");

  // wait for claw to be ready
  if (WaitUntil(::std::bind(&ShootAction::DoneSetupShot, this))) {
    return;
  }

  if (!IntakeOff()) return;

  // Make sure that we have the latest shooter status.
  control_loops::shooter_queue_group.status.FetchLatest();
  // Get the number of shots fired up to this point. This should not be updated
  // again for another few cycles.
  previous_shots_ = control_loops::shooter_queue_group.status->shots;
  // Shoot!
  if (!control_loops::shooter_queue_group.goal.MakeWithBuilder()
           .shot_power(control_loops::shooter_queue_group.goal->shot_power)
           .shot_requested(true)
           .unload_requested(false)
           .load_requested(false)
           .Send()) {
    LOG(WARNING, "sending shooter goal failed\n");
    return;
  }

  // wait for record of shot having been fired
  if (WaitUntil(::std::bind(&ShootAction::DoneShot, this))) return;

  if (!IntakeOff()) return;
}

bool ClawIsReady() {
  control_loops::claw_queue_group.goal.FetchLatest();

  bool ans =
      control_loops::claw_queue_group.status->zeroed &&
      (::std::abs(control_loops::claw_queue_group.status->bottom_velocity) <
       0.5) &&
      (::std::abs(control_loops::claw_queue_group.status->bottom -
                  control_loops::claw_queue_group.goal->bottom_angle) < 0.10) &&
      (::std::abs(control_loops::claw_queue_group.status->separation -
                  control_loops::claw_queue_group.goal->separation_angle) <
       0.4);
  if (!ans) {
    LOG(INFO, "Claw is %sready zeroed %d bottom_velocity %f bottom %f sep %f\n",
        ans ? "" : "not ", control_loops::claw_queue_group.status->zeroed,
        ::std::abs(control_loops::claw_queue_group.status->bottom_velocity),
        ::std::abs(control_loops::claw_queue_group.status->bottom -
                   control_loops::claw_queue_group.goal->bottom_angle),
        ::std::abs(control_loops::claw_queue_group.status->separation -
                   control_loops::claw_queue_group.goal->separation_angle));
  }
  return ans;
}

bool ShooterIsReady() {
  control_loops::shooter_queue_group.goal.FetchLatest();

  if (control_loops::shooter_queue_group.status->ready) {
    LOG(INFO, "Power error is %f - %f -> %f, ready %d\n",
        control_loops::shooter_queue_group.status->hard_stop_power,
        control_loops::shooter_queue_group.goal->shot_power,
        ::std::abs(control_loops::shooter_queue_group.status->hard_stop_power -
                   control_loops::shooter_queue_group.goal->shot_power),
        control_loops::shooter_queue_group.status->ready);
  }
  return (::std::abs(
              control_loops::shooter_queue_group.status->hard_stop_power -
              control_loops::shooter_queue_group.goal->shot_power) < 1.0) &&
         control_loops::shooter_queue_group.status->ready;
}

bool ShootAction::DoneSetupShot() {
  control_loops::shooter_queue_group.status.FetchAnother();
  control_loops::claw_queue_group.status.FetchAnother();
  // Make sure that both the shooter and claw have reached the necessary
  // states.
  if (ShooterIsReady() && ClawIsReady()) {
    LOG(INFO, "Claw and Shooter ready for shooting.\n");
    return true;
  }

  return false;
}

bool ShootAction::DonePreShotOpen() {
  control_loops::claw_queue_group.status.FetchAnother();
  if (control_loops::claw_queue_group.status->separation >
      kClawShootingSeparation) {
    LOG(INFO, "Opened up enough to shoot.\n");
    return true;
  }
  return false;
}

bool ShootAction::DoneShot() {
  control_loops::shooter_queue_group.status.FetchAnother();
  if (control_loops::shooter_queue_group.status->shots > previous_shots_) {
    LOG(INFO, "Shot succeeded!\n");
    return true;
  }
  return false;
}

::std::unique_ptr<TypedAction< ::frc971::actions::ShootActionQueueGroup>>
MakeShootAction() {
  return ::std::unique_ptr<
      TypedAction< ::frc971::actions::ShootActionQueueGroup>>(
      new TypedAction< ::frc971::actions::ShootActionQueueGroup>(
          &::frc971::actions::shoot_action));
}

}  // namespace actions
}  // namespace frc971

