#include "bbb/packet_finder.h"

#include <malloc.h>

#include <array>
#include <algorithm>

#include "gtest/gtest.h"

#include "aos/common/queue_testutils.h"
#include "aos/common/time.h"

#include "bbb/byte_io.h"

namespace bbb {
namespace testing {

class TestByteReader : public ByteReaderInterface {
 public:
  template <size_t bytes>
  TestByteReader(const ::std::array<uint8_t, bytes> &data)
      : data_(memalign(8, data.size())),
        data_size_(data.size()),
        bytes_left_(data.size()) {
    memcpy(data_, &data.at(0), data.size());
  }

  ~TestByteReader() {
    free(data_);
  }

  virtual ssize_t ReadBytes(uint8_t *dest, size_t max_bytes,
                            const ::aos::time::Time &/*timeout_time*/)
      override {
    size_t to_transfer = ::std::min(max_bytes, bytes_left_);
    if (to_transfer == 0) return -2;
    memcpy(dest, static_cast<const uint8_t *>(data_) + data_size_ - bytes_left_,
           to_transfer);
    bytes_left_ -= to_transfer;
    return to_transfer;
  }

 private:
  void *const data_;
  const size_t data_size_;
  size_t bytes_left_;
};

class PacketFinderTest : public ::testing::Test {
 public:
  void SetUp() override {
    ::aos::common::testing::EnableTestLogging();
  }

  template <typename Data, size_t N>
  void ReceivePackets(const Data &data, int packets,
                      ::std::array<int, N> expected_failures) {
    TestByteReader reader(data);
    PacketFinder packet_finder(&reader, 144);
    auto failure = expected_failures.begin();
    for (int i = 1; i < packets; ++i) {
      SCOPED_TRACE("packet " + ::std::to_string(i));
      bool expect_failure = false;
      if (failure != expected_failures.end() && *failure == i) {
        expect_failure = true;
        ++failure;
      }
      EXPECT_EQ(!expect_failure,
                packet_finder.ReadPacket(::aos::time::Time(0, 0)));
      if (expect_failure &&
          (failure == expected_failures.end() || i + 1 != *failure) &&
          i + 1 != packets) {
        int failures = 0;
        while (!packet_finder.ReadPacket(::aos::time::Time(0, 0))) {
          ++failures;
          SCOPED_TRACE("failure " + ::std::to_string(failures));
          ASSERT_LT(failures, 500);
        }
        i += 1;
      }
    }
    EXPECT_FALSE(packet_finder.ReadPacket(::aos::time::Time(0, 0)));
  }

  template <typename Data>
  void ReceivePackets(const Data &data, int packets) {
    // Not implemented as calling the overload with expected_failures because
    // ubsan doesn't like stdlibc++'s std::array (at least for now) and this is
    // really simple.
    TestByteReader reader(data);
    PacketFinder packet_finder(&reader, 144);
    for (int i = 1; i < packets; ++i) {
      SCOPED_TRACE("packet " + ::std::to_string(i));
      EXPECT_TRUE(packet_finder.ReadPacket(::aos::time::Time(0, 0)));
    }
    EXPECT_FALSE(packet_finder.ReadPacket(::aos::time::Time(0, 0)));
  }
};

static constexpr ::std::array<uint8_t, 896> kTestData1{
  {0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0xC0, 0x07, 0x01, 0x00, 0x02,
   0x00, 0x00, 0x00, 0x69, 0x3D, 0xE0, 0x03, 0x02, 0x00, 0x00, 0x00, 0x43, 0x5E,
   0x12, 0x16, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
   0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
   0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x01,
   0x00, 0x2A, 0x00, 0x00, 0x00, 0x10, 0x00, 0x2A, 0x00, 0x1B, 0x00, 0x4E, 0x01,
   0x72, 0x00, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0xF1, 0xFF, 0xFF,
   0xFF, 0x90, 0x01, 0x00, 0x20, 0x00, 0xED, 0x00, 0xE0, 0x01, 0x00, 0x00, 0x00,
   0x02, 0x00, 0x00, 0x00, 0x00, 0x38, 0x00, 0x40, 0x05, 0x00, 0x00, 0x00, 0x00,
   0x80, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x39, 0x00, 0x00, 0x00, 0x00, 0xED,
   0x00, 0xE0, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00,
   0x00, 0xFF, 0x8E, 0xFE, 0x9E, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
   0xD0, 0x07, 0x01, 0x00, 0x02, 0x00, 0x00, 0x00, 0xA1, 0x79, 0xE0, 0x03, 0x02,
   0x00, 0x00, 0x00, 0x43, 0x5E, 0x12, 0x16, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
   0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
   0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
   0x05, 0x00, 0x00, 0x00, 0x01, 0x00, 0x2A, 0x00, 0x00, 0x00, 0x10, 0x00, 0x29,
   0x00, 0x1B, 0x00, 0x50, 0x01, 0x72, 0x00, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00,
   0x00, 0x00, 0xF1, 0xFF, 0xFF, 0xFF, 0x90, 0x01, 0x00, 0x20, 0x00, 0xED, 0x00,
   0xE0, 0x01, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x38, 0x00, 0x40,
   0x03, 0x00, 0x00, 0x00, 0x00, 0xB0, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x06,
   0x00, 0x00, 0x00, 0xF1, 0xFF, 0xFF, 0xFF, 0x00, 0x38, 0x00, 0x40, 0x00, 0x38,
   0x00, 0x40, 0x82, 0x00, 0x00, 0x00, 0xE6, 0x6F, 0x7E, 0xC7, 0x00, 0x00, 0x00,
   0x00, 0x02, 0x00, 0x00, 0x00, 0xE0, 0x07, 0x01, 0x00, 0x02, 0x00, 0x00, 0x00,
   0xDA, 0xB5, 0xE0, 0x03, 0x02, 0x00, 0x00, 0x00, 0x43, 0x5E, 0x12, 0x16, 0x01,
   0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
   0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
   0x00, 0x01, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x02, 0x00, 0x2B, 0x00,
   0x00, 0x00, 0x10, 0x00, 0x2A, 0x00, 0x1B, 0x00, 0x50, 0x01, 0x72, 0x00, 0x01,
   0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0xF1, 0xFF, 0xFF, 0xFF, 0x90, 0x01,
   0x00, 0x20, 0x00, 0xED, 0x00, 0xE0, 0x01, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00,
   0x00, 0x00, 0x38, 0x00, 0x40, 0x05, 0x00, 0x00, 0x00, 0x00, 0xD0, 0x00, 0x00,
   0x00, 0x10, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0xF1, 0xFF, 0xFF, 0xFF, 0x00,
   0x38, 0x00, 0x40, 0x00, 0x38, 0x00, 0x40, 0x80, 0x00, 0x00, 0x00, 0xDD, 0xD1,
   0x5D, 0x8E, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0xF0, 0x07, 0x01,
   0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0xF2, 0xE0, 0x03, 0x02, 0x00, 0x00, 0x00,
   0x43, 0x5E, 0x12, 0x16, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
   0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
   0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00,
   0x00, 0x02, 0x00, 0x2B, 0x00, 0x00, 0x00, 0x10, 0x00, 0x29, 0x00, 0x1B, 0x00,
   0x50, 0x01, 0x72, 0x00, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0xF1,
   0xFF, 0xFF, 0xFF, 0x90, 0x01, 0x00, 0x20, 0x00, 0xED, 0x00, 0xE0, 0x01, 0x00,
   0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x38, 0x00, 0x40, 0x01, 0x00, 0x00,
   0x00, 0x00, 0x90, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00,
   0x00, 0xED, 0x00, 0xE0, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02,
   0x00, 0x00, 0x00, 0x68, 0x26, 0xC4, 0xF1, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00,
   0x00, 0x00, 0x00, 0x08, 0x01, 0x00, 0x02, 0x00, 0x00, 0x00, 0x4B, 0x2E, 0xE1,
   0x03, 0x02, 0x00, 0x00, 0x00, 0x43, 0x5E, 0x12, 0x16, 0x01, 0x00, 0x00, 0x00,
   0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
   0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
   0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x01, 0x00, 0x2A, 0x00, 0x00, 0x00, 0x10,
   0x00, 0x2A, 0x00, 0x1C, 0x00, 0x50, 0x01, 0x72, 0x00, 0x01, 0x00, 0x00, 0x00,
   0x04, 0x00, 0x00, 0x00, 0xF1, 0xFF, 0xFF, 0xFF, 0x90, 0x01, 0x00, 0x20, 0x00,
   0xED, 0x00, 0xE0, 0x01, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x38,
   0x00, 0x40, 0x04, 0x00, 0x00, 0x00, 0x00, 0xC0, 0x00, 0x00, 0x00, 0x10, 0x00,
   0x00, 0x06, 0x00, 0x00, 0x00, 0xF1, 0xFF, 0xFF, 0xFF, 0x00, 0x38, 0x00, 0x40,
   0x00, 0x38, 0x00, 0x40, 0x00, 0x38, 0x00, 0x40, 0xAD, 0x8B, 0x6A, 0x10, 0x00,
   0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x10, 0x08, 0x01, 0x00, 0x02, 0x00,
   0x00, 0x00, 0x84, 0x6A, 0xE1, 0x03, 0x02, 0x00, 0x00, 0x00, 0x43, 0x5E, 0x12,
   0x16, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
   0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01,
   0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x02, 0x00,
   0x2A, 0x00, 0x00, 0x00, 0x10, 0x00, 0x29, 0x00, 0x1B, 0x00, 0x4E, 0x01, 0x72,
   0x00, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0xF1, 0xFF, 0xFF, 0xFF,
   0x90, 0x01, 0x00, 0x20, 0x00, 0xED, 0x00, 0xE0, 0x01, 0x00, 0x00, 0x00, 0x07,
   0x00, 0x00, 0x00, 0x00, 0x38, 0x00, 0x40, 0x06, 0x00, 0x00, 0x00, 0x00, 0xE0,
   0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x39, 0x00, 0x00, 0x00, 0x00, 0xED, 0x00,
   0xE0, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
   0xFC, 0x94, 0x39, 0xF5, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00}
};

// This data has some leading 0s in the data part of at least one of the packets
// that used to confuse the code.
TEST_F(PacketFinderTest, LeadingZeros) {
  ReceivePackets(kTestData1, 7);
}

// Tests to make sure that dropping any byte will result in missing that packet
// and no future ones.
TEST_F(PacketFinderTest, DropAnyByte) {
  static constexpr auto kTestData = kTestData1;
  for (int i = 0; i < static_cast<int>(kTestData.size()); ++i) {
    SCOPED_TRACE("dropping byte " + ::std::to_string(i));
    ::std::array<uint8_t, kTestData.size() - 1> data;
    ::std::copy(kTestData.begin(), kTestData.begin() + i, data.begin());
    ::std::copy(kTestData.begin() + i + 1, kTestData.end(),
                data.begin() + i);
    ReceivePackets(data, 7, ::std::array<int, 1>{{i / 148 + 1}});
  }
}

// Tests to make sure that bitwise-NOTing any byte will result in missing that
// packet and no future ones.
TEST_F(PacketFinderTest, InvertAnyByte) {
  static constexpr auto kTestData = kTestData1;
  for (int i = 0; i < static_cast<int>(kTestData.size()); ++i) {
    SCOPED_TRACE("inverting byte " + ::std::to_string(i));
    ::std::array<uint8_t, kTestData.size()> data;
    ::std::copy(kTestData.begin(), kTestData.end(), data.begin());
    data.at(i) ^= 0xFF;
    ReceivePackets(data, 7, ::std::array<int, 1>{{i / 148 + 1}});
  }
}

}  // namespace testing
}  // namespace bbb
