#ifndef BBB_CAPE_SRC_BBB_PACKET_FINDER_H_
#define BBB_CAPE_SRC_BBB_PACKET_FINDER_H_

#include <stdint.h>
#include <string.h>

#include "aos/common/logging/logging.h"
#include "aos/common/time.h"
#include "aos/common/macros.h"
#include "aos/common/util/log_interval.h"

namespace bbb {

class ByteReaderInterface;

class PacketFinder {
 public:
  // *reader has to stay alive for the entire lifetime of this object but this
  // object does not take ownership.
  PacketFinder(ByteReaderInterface *reader, size_t packet_size);
  ~PacketFinder();

  // Returns true if it succeeds or false if it gets an I/O error (or timeout)
  // first.
  // timeout_time is when this method should give up trying to read a packet and
  // return false.
  bool ReadPacket(const ::aos::time::Time &timeout_time);

  // Gets a reference to the received packet.
  // The most recent call to ReadPacket() must have returned true or the data
  // pointed to is undefined.
  template <typename T>
  const T *get_packet() {
    static_assert(alignof(T) <= alignof(AlignedChar),
                  "We need to align our data better.");
    CHECK(sizeof(T) <= packet_size_ - 8);
    return reinterpret_cast<const T *>(unstuffed_data_);
  }

 private:
  // We have 64-bit ints in some of our data.
  typedef char __attribute__((aligned(8))) AlignedChar;

  static const int kZeros = 4;

  // Reads bytes until there are 4 zeros and then fills up buf_.
  // Returns true if it finds one or false if it gets an I/O error first or the
  // packet is invalid in some way.
  bool FindPacket(const ::aos::time::Time &timeout_time);

  // Processes a packet currently in buf_ and leaves the result in
  // unstuffed_data_.
  // Returns true if it succeeds or false if there was something wrong with the
  // data.
  bool ProcessPacket();

  // Avoids issues with accessing elements of buf_ that are technically not
  // aligned correctly.
  uint8_t buf(size_t index) const {
    uint8_t r;
    memcpy(&r, &buf_[index], 1);
    return r;
  }

  ByteReaderInterface *const reader_;
  const size_t packet_size_;

  AlignedChar *const buf_;
  AlignedChar *const unstuffed_data_;

  // How many bytes of the packet we've read in (or -1 if we don't know where
  // the packet is).
  int packet_bytes_ = -1;

  // Whether we've increased the priority of the IRQ yet.
  bool irq_priority_increased_ = false;

  typedef ::aos::util::SimpleLogInterval SimpleLogInterval;
  static constexpr ::aos::time::Time kDebugLogInterval =
      ::aos::time::Time::InSeconds(0.3);
  SimpleLogInterval invalid_packet_ =
      SimpleLogInterval(kDebugLogInterval, WARNING, "invalid packet");
  SimpleLogInterval bad_checksum_ =
      SimpleLogInterval(kDebugLogInterval, WARNING, "bad checksum");

  DISALLOW_COPY_AND_ASSIGN(PacketFinder);
};

}  // namespace bbb

#endif
