#ifndef AOS_COMMON_QUEUE_TYPES_H_
#define AOS_COMMON_QUEUE_TYPES_H_

#include <sys/types.h>
#include <stdint.h>
#include <string.h>

#include <initializer_list>
#include <string>

#include "aos/common/macros.h"
#include "aos/common/byteorder.h"

namespace aos {

// The type IDs this uses are 2 parts: a 16 bit size and a 16 bit hash. Sizes
// for primitive types are stored with 8192 (0x2000) added.
//
// Serializing/deserializing includes all of the fields too.
struct MessageType {
  struct Field {
    // The type ID for the type of this field.
    uint32_t type;
    ::std::string name;
  };

  // Takes ownership of all the Field pointers in fields_initializer.
  MessageType(size_t super_size, uint32_t id, const ::std::string &name,
              ::std::initializer_list<const Field *> fields_initializer)
      : super_size(super_size), id(id), name(name) {
    number_fields = fields_initializer.size();
    fields = new const Field *[number_fields];
    int i = 0;
    for (const Field *c : fields_initializer) {
      fields[i++] = c;
    }
  }

  ~MessageType() {
    for (int i = 0; i < number_fields; ++i) {
      delete fields[i];
    }
    delete[] fields;
  }

  // Returns -1 if max_bytes is too small.
  ssize_t Serialize(char *buffer, size_t max_bytes) const;
  // bytes should start out as the number of bytes available in buffer and gets
  // reduced by the number actually read before returning.
  // Returns a new instance allocated with new or nullptr for error.
  static MessageType *Deserialize(const char *buffer, size_t *bytes);

  static bool IsPrimitive(uint32_t type_id) {
    return (type_id & 0x2000) != 0;
  }

  static unsigned int Sizeof(uint32_t type_id) {
    if (IsPrimitive(type_id)) {
      return (type_id & 0xFFFF) - 0x2000;
    } else {
      return type_id & 0xFFFF;
    }
  }

  // How many (serialized) bytes the superclass takes up.
  uint16_t super_size;
  // The type ID for this.
  uint32_t id;
  ::std::string name;

  uint16_t number_fields;
  const Field **fields;

 private:
  // Internal constructor for Deserialize to use.
  MessageType(uint16_t super_size, uint32_t id, ::std::string &&name,
              uint16_t number_fields, Field **fields)
      : super_size(super_size),
        id(id),
        name(name),
        number_fields(number_fields),
        fields(const_cast<const Field **>(fields)) {
    for (int i = 0; i < number_fields; ++i) {
      fields[i] = new Field();
    }
  }

  DISALLOW_COPY_AND_ASSIGN(MessageType);
};

// The arguments are the same for both of these.
//
// output is where to write the text representation ('\0'-terminated
// human-readable string).
// output_bytes should point to the number of bytes available to write in
// output. It will be reduced by the number of bytes that were actually written.
// input is where to read the data in from (in network byte order, aka from
// Serialize).
// input_bytes should point to the number of bytes available to read from input.
// It will be reduced by the number of bytes that were actually read.
// type is which type to print.
// Returns true for success and false for not.
//
// Prints the value from 1 primitive message field into output.
// The implementation of this is generated by the ruby code.
// Does not include a trailing '\0' in what it subtracts from *output_bytes.
bool PrintField(char *output, size_t *output_bytes, const void *input,
                size_t *input_bytes, uint32_t type)
    __attribute__((warn_unused_result));
// Prints a whole message into output.
// This calls itself recursively and PrintField to print out the whole thing.
bool PrintMessage(char *output, size_t *output_bytes, const void *input,
                  size_t *input_bytes, const MessageType &type)
    __attribute__((warn_unused_result));
// Calls PrintField to print out a matrix of values.
bool PrintMatrix(char *output, size_t *output_bytes, const void *input,
                 uint32_t type, int rows, int cols);

// "Serializes" a matrix (basically just converts to network byte order). The
// result can be passed to PrintMatrix.
void SerializeMatrix(int type_id, void *output_void, const void *input_void,
                     int rows, int cols);

// Implements a cache of types which generally works per-process but can (when
// instructed) put a type in shared memory which other processes will
// automatically be able to retrieve.
// All of these functions are thread-safe.
namespace type_cache {

// Makes sure a type is in the type cache. This will store a reference to type.
// The types of any non-primitive fields of type must already be added.
void Add(const MessageType &type);
// Retrieves a type from the type cache or shm. LOG(FATAL)s if it can't find it.
const MessageType &Get(uint32_t type_id);
// Makes sure a type is in the list in shm. Add must have already been called
// for type.
// Also adds the types of any non-primitive fields of type.
void AddShm(uint32_t type_id);

}  // namespace type_cache
}  // namespace aos

#endif  // AOS_COMMON_QUEUE_TYPES_H_
