(*:Name: Transform *)

(*:Title: Transform *)

(*:Author: Tom Wickham-Jones*)

(*:Package Version: 1.0 *)

(*:Mathematica Version: 2.2 *)


(*:Summary:
	This package provides functions to carry out homogeneous
	affine transformations on two and three-dimensional graphics
	primitives.
*)

(*:History:
	Created winter 1994 by Tom Wickham-Jones.

	This package is described in the book
	Mathematica Graphics: Techniques and Applications.
	Tom Wickham-Jones, TELOS/Springer-Verlag 1994.

*)

If[ Context[ HomogeneousMultiply] =!= "ExtendGraphics`Transform`",
		Remove[ HomogeneousMultiply]]

If[ Context[ HomogeneousMatrix] =!= "ExtendGraphics`Transform`",
		Remove[ HomogeneousMatrix]]

If[ Context[ HomogeneousTranslation] =!= "ExtendGraphics`Transform`",
		Remove[ HomogeneousTranslation]]

If[ Context[ HomogeneousRotation] =!= "ExtendGraphics`Transform`",
		Remove[ HomogeneousRotation]]

If[ Context[ HomogeneousScale] =!= "ExtendGraphics`Transform`",
		Remove[ HomogeneousScale]]

If[ Context[ HomogeneousShear] =!= "ExtendGraphics`Transform`",
		Remove[ HomogeneousShear]]

If[ Context[ HomogeneousTransform] =!= "ExtendGraphics`Transform`",
		Remove[ HomogeneousTransform]]

  

BeginPackage[ "ExtendGraphics`Transform`",
				"Geometry`Rotations`"]

HomogeneousMultiply::usage =
	"HomogeneousMultiply[ mat, vec] multiplies the vector vec by the
	 homogeneous transformation matrix mat."


HomogeneousMatrix::usage = 
	"HomogeneousMatrix[ mat] converts a matrix mat to a homogeneous
	 transformation matrix."
	
	
HomogeneousTranslation::usage = 
	"HomogeneousTranslation[ tran] returns the homogeneous
	 matrix that represents the translation tran."

	
HomogeneousRotation::usage = 
	"HomogeneousRotation[ t] returns the homogeneous matrix
	 that represents the two-dimensional rotation t.
	 HomogeneousRotation[ {phi, theta, psi}] returns the homogeneous
	 matrix that represents the three-dimensional rotation 
	 specified by Euler angles {phi, theta, psi}."


HomogeneousScale::usage = 
	"HomogeneousScale[ svec] returns the homogeneous
	 matrix that represents the scaling svec."
	
	
HomogeneousShear::usage = 
	"HomogeneousShear[ {sx, sy}] returns the homogeneous matrix
	 that represents a two-dimensional of sx along the x-axis and
	 sy along the y-axis. HomogeneousShear[ {{Xy, Xz}, {Yx, Yz}, {Zx, Zy}}]
	 returns the homogeneous matrix that represents the three-dimensional
	 shear of {Xy, Xz} in the yz-plane, {Yx, Yz} in the xz plane and
	 {Zx, Zy} in the xy-plane."
	 
	
HomogeneousTransform::usage = 
	"HomogeneousTransform[ mat, g] applies the transformation specified
	 by mat to the graphics object g."

Begin[ "`Private`"]


HomogeneousMultiply[ mat_, vec_] :=
	Drop[ mat.Append[ vec,1], -1]


HomogeneousTranslation[ {tx_, ty_}] :=
    {{1,0,tx}, {0,1,ty}, {0,0,1}}


HomogeneousTranslation[ {tx_, ty_, tz_}] :=
    {{1,0,0,tx}, {0,1,0,ty}, {0,0,1,tz},{0,0,0,1}}


HomogeneousRotation[ t_] :=
    Append[ 
      Map[ Append[#, 0]&, RotationMatrix2D[t]], 
      {0,0,1}]

HomogeneousRotation[ {phi_,theta_,psi_}] :=
    Append[ 
      Map[ Append[#, 0]&, RotationMatrix3D[ phi, theta, psi]], 
      {0,0,0,1}]

HomogeneousScale[ {sx_, sy_}] :=
    Append[ 
      Map[ Append[#, 0]&, {{sx, 0}, {0, sy}}], 
      {0,0,1}]

HomogeneousScale[ {sx_, sy_, sz_}] :=
    {{sx,0,0,0}, {0,sy,0,0}, {0,0,sz,0},{0,0,0,1}} 


HomogeneousShear[ {sx_, sy_}] :=
    {{1, sx, 0}, {sy, 1, 0}, {0,0,1}} 


HomogeneousShear[ {{Xy_, Xz_}, {Yx_, Yz_}, {Zx_, Zy_}}] :=
    {{1,Yx,Zx,0}, {Xy,1,Zy,0}, {Xz,Yz,1,0},{0,0,0,1}} 


HomogeneousTransform[ 
		mat_ /; Dimensions[ mat] === {3,3}, 
		Graphics[ prims_, opts___]] :=
	Graphics[ HomogeneousTransform[ mat, prims], opts]
	
HomogeneousTransform[ 
		mat_ /; Dimensions[ mat] === {4,4}, 
		Graphics3D[ prims_, opts___]] :=
	Graphics3D[ HomogeneousTransform[ mat, prims], opts]
	
HomogeneousTransform[ mat_, l_List] :=
	Map[ HomogeneousTransform[ mat, #]&, l]
	
HomogeneousTransform[ mat_, Line[ pts_]] :=
	Line[ Map[ HomogeneousMultiply[ mat, #]&, pts]]
	
HomogeneousTransform[ mat_, Polygon[ pts_]] :=
	Polygon[ Map[ HomogeneousMultiply[ mat, #]&, pts]]

HomogeneousTransform[ mat_, Point[ pt_]] :=
	Point[ HomogeneousMultiply[ mat, pt]]
	
HomogeneousTransform[ mat_, Text[ t_, pt_, rest___]] :=
	Text[ t, HomogeneousMultiply[ mat, pt], rest]
	


End[]

EndPackage[]


(*:Examples:

<<ExtendGraphics`Transform`


p = ParametricPlot3D[ {Sin[t] Sin[p], Sin[t] Cos[p], Cos[t]},
                {t,0,Pi},{p,0,2Pi}]


mat = HomogeneousTranslate[ {1,1,1}]
		
Show[ HomogeneousTransform[ mat, p]]

shear = HomogeneousShear[ {{0,0}, {0,0}, {1,1}}]


Show[ HomogeneousTransform[ shear, p]]

shear = HomogeneousShear[ {{1,1}, {0,0}, {0,0}}]


Show[ HomogeneousTransform[ shear, p]]

shear = HomogeneousShear[ {{0,0}, {1,1}, {0,0}}]


Show[ HomogeneousTransform[ shear, p]]

p = ParametricPlot3D[ {Sin[z] Sin[p], Sin[z] Cos[p], z},
                {z,-Pi,Pi},{p,0,2Pi},PlotPoints -> 21]


rot = HomogeneousRotation[ {0,Pi/2,0}]

Show[ HomogeneousTransform[ rot, p]]

		
*)

