#ifndef AOS_STARTER_STARTERD_LIB_H_
#define AOS_STARTER_STARTERD_LIB_H_

#include <sys/signalfd.h>
#include <sys/wait.h>

#include <csignal>
#include <cstdio>
#include <string>
#include <unordered_map>
#include <vector>

#include "aos/configuration.h"
#include "aos/events/shm_event_loop.h"
#include "aos/ipc_lib/signalfd.h"
#include "aos/macros.h"
#include "aos/starter/starter_generated.h"
#include "aos/starter/starter_rpc_generated.h"
#include "aos/starter/subprocess.h"

namespace aos {
namespace starter {

const aos::Channel *StatusChannelForNode(const aos::Configuration *config,
                                         const aos::Node *node);
const aos::Channel *StarterRpcChannelForNode(const aos::Configuration *config,
                                             const aos::Node *node);

class Starter {
 public:
  Starter(const aos::Configuration *event_loop_config);

  // Inserts a new application from config. Returns the inserted application if
  // it was successful, otherwise nullptr if an application already exists
  // with the given name.
  Application *AddApplication(const aos::Application *application);

  // Runs the event loop and starts all applications
  void Run();

  void Cleanup();

 private:
  // Signals which indicate starter has died
  static const inline std::vector<int> kStarterDeath = {
      SIGHUP,  SIGINT,  SIGQUIT, SIGILL, SIGABRT, SIGFPE,
      SIGSEGV, SIGPIPE, SIGTERM, SIGBUS, SIGXCPU};

  void OnSignal(signalfd_siginfo signal);
  void HandleStarterRpc(const StarterRpc &command);

  // Sends the Status message if it wouldn't exceed the rate limit.
  void MaybeSendStatus();

  void SendStatus();

  const std::string config_path_;
  const aos::Configuration *config_msg_;

  aos::ShmEventLoop event_loop_;
  aos::Sender<aos::starter::Status> status_sender_;
  aos::TimerHandler *status_timer_;
  aos::TimerHandler *cleanup_timer_;

  int status_count_ = 0;
  const int max_status_count_;

  std::unordered_map<std::string, Application> applications_;

  // Set to true on cleanup to block rpc commands and ensure cleanup only
  // happens once.
  bool exiting_ = false;

  SignalListener listener_;

  DISALLOW_COPY_AND_ASSIGN(Starter);
};

}  // namespace starter
}  // namespace aos

#endif  // AOS_STARTER_STARTERD_LIB_H_
