/*

 Author: Tom Wickham-Jones WRI (twj@wri.com).
	
 A MathLink application to carry out Delaunay 
 triangulations in the plane.

 The application is described in the book
 Mathematica Graphics: Techniques and Applications.
 Tom Wickham-Jones, TELOS/Springer-Verlag 1994.

 To build on Unix systems enter
 
 	mcc -o delaunay delaunay.tm

 to the shell.
 
 This presupposes an ANSI compiler called cc.

 If you have an ANSI compiler that is not called cc you
 must edit the mcc script and make the necessary change.
 (Future versions of mcc will do this with a variable).
 
 If you do not have an ANSI compiler then you must compile
 it without prototypes:
 
	mcc -o delaunay -DNOPROTO delaunay.tm


  To build this on the Macintosh or Windows consult your MathLink
  documentation.   There are Macintosh and Windows binaries
  available.
  
*/

:Begin:
:Function:       delaunay
:Pattern:        ExtendGraphics`Delaunay`Private`DelaunayL[d_List /; MatrixQ[ d, NumberQ] && Length[First[d]] === 2]
:Arguments:      { d }
:ArgumentTypes:  {Manual}
:ReturnType:     Manual 
:End:
#include <math.h>
#include "mathlink.h"
#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <stddef.h>

#ifdef NOPROTO
#define PROTO 0
#endif

#ifndef PROTO
#define PROTO 1
#endif


/*
  For all these data structures
  Nexts go CCW and Prevs go CW
*/

struct pointstruct
  {
  int	trinum;
  int	hullpt;
  int	triused;
  int	encode;
  double x, y;
  };

typedef struct pointstruct spoint;
typedef struct pointstruct * point;


struct hullstruct
  {
    int 				elem;
	int					collinear;
    struct hullstruct 	*end1;
    struct hullstruct 	*end2;
    struct hullstruct 	*next;
    struct hullstruct 	*prev;
  } ;

typedef struct hullstruct 	*hull;

#define HullCollinear( h)	h->collinear
#define HullElem( h)		h->elem
#define HullNext( h)		h->next
#define HullEnd1( h)		h->end1
#define HullEnd2( h)		h->end2
#define HullPrev( h)		h->prev
#define	HullTest( h)	((HullNext( HullPrev(h)) == h) && \
			 (HullPrev( HullNext(h)) == h))

#define HullJoin( hl, hr)	(HullNext( hl) = hr, HullPrev( hr) = hl)

#define HullPt( i)	pnts[i].hullpt
#define TriNum( i)	pnts[i].trinum
#define TriUsed( i)	pnts[i].triused
#define Encode( i)	pnts[i].encode
#define Coord( i)	pnts[i]
#define XCoord( i)	pnts[i].x
#define YCoord( i)	pnts[i].y
#define XCoordPt( pt)	(pt).x
#define YCoordPt( pt)	(pt).y

struct jactstruct
  {
    int 		jt_elem;
    int 		jt_len;
    double		jt_angle;
    struct jactstruct 	*jt_next;
    struct jactstruct	*jt_prev;
  };
  
typedef struct jactstruct *jact;

#define JactElem( h)	h->jt_elem
#define JactAngle(h)	h->jt_angle
#define JactLen(h)		h->jt_len
#define JactNext( h)	h->jt_next
#define JactPrev( h)	h->jt_prev
#define	JactTest( j)	((JactNext( JactPrev(j)) == j) && \
			 (JactPrev( JactNext(j)) == j))
#define JactJoin( jl, jr)	(JactNext( jl) = jr, JactPrev( jr) = jl)


struct tristruct
  {
    int			pt[3];
    struct tristruct	*next;
  };

typedef struct tristruct *tri;

#define TriA(t)		t->pt[0]
#define TriB(t)		t->pt[1]
#define TriC(t)		t->pt[2]
#define TriPt(t, i)	t->pt[i]
#define TriNext(t)	t->next



point 	pnts 		= NULL;
jact	* jactpoint = NULL;
int 	dlen;


#define	TRUE		1
#define	FALSE		0
#define MPI2		6.2831853071795864769
#define	EPS		1.e-15
#define	LOW		0
#define UPP		1
#define	RIGHT		1
#define	LEFT		-1
#define	COLLINEAR	0




int 
#if PROTO
	ordered( const void *p11, const void *p22)
#else
	ordered( p11, p22) spoint *p11, *p22;
#endif
{
  
  spoint *p1 = (spoint *) p11;
  spoint *p2 = (spoint *) p22;
  
  if ( XCoordPt( *p1) < XCoordPt( *p2))
  	return -1;
  
  if ( XCoordPt( *p1) > XCoordPt( *p2))
  	return 1;
  
  if ( YCoordPt( *p1) < YCoordPt( *p2))
  	return -1;

  if ( YCoordPt( *p1) > YCoordPt( *p2))
  	return 1;
	
  assert( FALSE);
  	return 0;
}

static tri
#if PROTO
	cons_tri( void)
#else
	cons_tri()
#endif
{
  tri t;
  
  t = (tri) malloc( sizeof(struct tristruct));
  
  TriNext(t) = NULL;
  return t;
}

static void
#if PROTO
	des_tri( tri t)
#else
	des_tri( t) tri t;
#endif
{

  free( t);
}

static void 
#if PROTO
	cons_data( int len)
#else
	cons_data( len) int len;
#endif
{
	int i;
  
	pnts = (point) malloc( sizeof( struct pointstruct)*len);
	jactpoint = (jact *)malloc( sizeof( jact)*len);
	for ( i = 0; i < len; i++) 
		{
		Encode(i) = i;
		TriNum(i) = 0;
		HullPt(i) = FALSE;
		TriUsed(i) = 0;
		jactpoint[i] = NULL;
		}
}

static void
#if PROTO
	des_data( hull h, int len)
#else
	des_data( h, len) hull h; int len;
#endif
{
  hull h1;
  jact j, j1;
  int i;
  
  assert( HullTest( h));
  HullNext( HullPrev(h)) = NULL;
  while( h != NULL) {
  	h1 = h;
  	h = HullNext( h);
	free( h1);
  }	
  
  for ( i = 0; i < len; i++) {
  	j = jactpoint[i];
	assert( JactTest( j));
	JactNext( JactPrev(j)) = NULL;
  	while( j != NULL) {
  		j1 = j;
  		j = JactNext( j);
		free( j1);
	}
  }	

  if (jactpoint != NULL)
  	free( jactpoint);

  if ( pnts != NULL)
  	free( pnts);
}

static hull
#if PROTO
	cons_hull( int a)
#else
	cons_hull( a) int a;
#endif
{
  hull h;
  
  h = (hull)malloc( sizeof(struct hullstruct)) ;
  HullCollinear(h) = FALSE;
  HullElem( h) = a;
  return h;
}

static void
#if PROTO
	des_hull( hull h)
#else
	des_hull( h) hull h;
#endif
{

  free( h);
}

static int
#if PROTO
	len_hull( hull h)
#else
	len_hull( h) hull h;
#endif
{
  	hull hn = HullNext( h);
  	int len = 1;
  
  	while( hn != h) 
		{
  		hn = HullNext( hn);
		len++;
  	}
  	return len;
}


static void
#if PROTO
	des_jact( jact j)
#else
	des_jact( j) jact j;
#endif
{

  free( j);
}


static jact
#if PROTO
	cons_jact( int a)
#else
	cons_jact( a) int a;
#endif
{
  jact j;
  
  j = (jact) malloc( sizeof( struct jactstruct));
  JactElem( j) = a;
  return j;
}

#ifdef DEBUG
static void
#if PROTO
	p_hull( hull h)
#else
	p_hull( h) hull h;
#endif
{
  hull h1 = HullNext(h);
  
  printf(" %d", HullElem( h));
  while( h1 != h) {
  	printf( " %d", HullElem( h1));
	h1 = HullNext(h1);
  }
  printf( "\n");
}

static void
#if PROTO
	p_jactpt( int ij)
#else
	p_jactpt( ij) int ij;
#endif
{
	jact j = jactpoint[ ij];

	jact j1 = JactNext(j);
  
	printf(" %d", JactElem( j));
	while( j1 != j) 
		{
		printf( " %d", JactElem( j1));
		j1 = JactNext(j1);
		}
		printf( "\n");
}

static void 
#if PROTO
	p_jact( jact j)
#else
	p_jact( j) jact j;
#endif
{

	jact j1 = JactNext(j);
  
	printf(" %d", JactElem( j));
	while( j1 != j) 
		{
		printf( " %d", JactElem( j1));
		j1 = JactNext(j1);
		}
		printf( "\n");
}

static void 
#if PROTO
	check_jact(void)
#else
	check_jact()
#endif
{
  int i, cnt;
  jact j, j1;
  
  for ( i = 0; i < dlen; i++) {
  	j = jactpoint[i];
	if ( j == NULL)
		continue;
	j1 = JactNext( j);
	assert( JactTest(j));
	cnt = 0;
	while( j1 != j) {
		assert( JactTest(j1));
		cnt++;
		assert( cnt <= dlen);
		j1 = JactNext( j1);
	}
  }

}
#endif

static double
#if PROTO
	norm( int a, int b)
#else
	norm( a, b) int a, b;
#endif
{
  	double res, b1, b2;

	b1 = (XCoord( b) - XCoord(a));
	b1 = b1*b1;
	
	b2 = (YCoord( b) - YCoord(a));
	b2 = b2*b2;
	
	res = sqrt( b1*b1 + b2*b2);
	
	assert( res != 0.0);
	
	return res;
}



/*
 If c is right of the line from a to b return True.
 The test is made so that if they are close to collinear
 False is returned.
*/

static int
#if PROTO
  	side_of_line( int a, int b, int c)
#else
  	side_of_line( a, b, c) int a, b, c;
#endif
{
  	double bnorm, cnorm, ux, uy, vx, vy, test;

	
	bnorm = norm( a, b);
	
	cnorm = norm( a, c);
	
	ux = (XCoord(b) - XCoord( a))/bnorm;	
	uy = (YCoord(b) - YCoord( a))/bnorm;	
	
	vx = (XCoord(c) - XCoord( a))/cnorm;	
	vy = (YCoord(c) - YCoord( a))/cnorm;	
	
	
	test = ux*vy - uy*vx;

	if ( test > EPS)
		return LEFT;
	else if ( test < -EPS)
		return RIGHT;
	
  	return COLLINEAR;
}

/*
 If c is right of the line from a to b return True.
 The test is made so that if they are close to collinear
 False is returned.
*/

static int
#if PROTO
	right_of_line( int a, int b, int c)
#else
	right_of_line( a, b, c) int a, b, c;
#endif
{
  return (side_of_line( a, b, c) == RIGHT);
}


static hull
#if PROTO
	new_hull( int a, int b, int c)
#else
	new_hull( a, b, c) int a, b, c;
#endif
{
  hull h1, h2, h3;
  
  h1 = cons_hull(a);
  h2 = cons_hull(b);
  h3 = cons_hull(c);
  
  HullJoin( h1, h2);
  HullJoin( h2, h3);
  HullJoin( h3, h1);
  return h1;
}

/*
  This is for collinear points.
  The hull is recorded by the start and end points.
  This decision is arbitary but consistently followed
  through the algorithm.  When the hull expands to non-collinear
  everything will work.
*/
static hull
#if PROTO
	new_hull4( int a, int b, int c)
#else
	new_hull4( a, b, c) int a, b, c;
#endif
{
  hull h1, h2, h3, h4;
  
  h1 = cons_hull(a);
  h2 = cons_hull(b);
  h3 = cons_hull(c);
  h4 = cons_hull(b);
  
  HullCollinear( h1) = TRUE;
  HullCollinear( h2) = TRUE;
  HullCollinear( h3) = TRUE;
  HullCollinear( h4) = TRUE;
  
  HullEnd1( h1) = h1;
  HullEnd2( h1) = h3;
  HullEnd1( h2) = h1;
  HullEnd2( h2) = h3;
  HullEnd1( h3) = h1;
  HullEnd2( h3) = h3;
  HullEnd1( h4) = h1;
  HullEnd2( h4) = h3;
  
  HullJoin( h1, h2);
  HullJoin( h2, h3);
  HullJoin( h3, h4);
  HullJoin( h4, h1);

  return h1;
}

/*
  jact's should go CCW
  angs should become bigger
*/
static double 
#if PROTO
	fix_atan2( double y, double x)
#else
	fix_atan2( y, x) double y, x;
#endif
{
  double ang;
  
  ang = atan2( y, x);
  
  if (ang < 0.0)
  	return MPI2 + ang;
  return ang;
}

static double
#if PROTO
	dist2(int l, int r)
#else
	dist2(l, r) int l, r;
#endif
{
  double x, y;
  
  x = XCoord( l) - XCoord( r);
  y = YCoord( l) - YCoord( r);
  return x*x + y*y;
}


static int
#if PROTO
	first_smaller( int pt, jact j1, jact j2)
#else
	first_smaller(pt, j1, j2) int pt; jact j1, j2;
#endif
{

	if ( JactAngle( j1) == JactAngle( j2))
		{
		if ( dist2( pt, JactElem( j1)) > dist2( pt, JactElem( j2)))
			return TRUE;
		return FALSE;
		}
		
	if ( JactAngle( j1) < JactAngle( j2))
		return TRUE;
	
	return FALSE;
}


static void
#if PROTO
	jact_join(jact jp, jact j, jact jn)
#else
	jact_join(jp, j, jn) jact jp, j, jn;
#endif
{

  JactNext( jp) = j;
  JactPrev(  j) = jp;
  JactNext(  j) = jn;
  JactPrev( jn) = j;
}


/*
  insert r into the adjacency list (jact) of l.
  If this is the first point or the second point
  then care.
  
  The list is kept in a sorted fashion with minimum at
  j = jactpoint[l],  JactNext( j) will be bigger.
  The biggest will be JactPrev( j). 

  The size depends upon the angle, if there is a tie 
  then the outter-most goes first.
  We go CCW out-to-in.
  
*/
static void
#if PROTO
	jactinsert1(int l, int r)
#else
	jactinsert1(l, r) int l, r;
#endif
{
	jact j, j1, jp, jnew;
	int i, len, test;
	
  
	jnew = cons_jact(r);
	JactAngle(jnew) = fix_atan2( 
						YCoord( r) - YCoord( l),
						XCoord( r) - XCoord( l)
					);

	j = jactpoint[ l];

  	if ( j == NULL)    /* special case length 0 */
		{
  		jactpoint[l] = jnew;
		JactLen( jnew) = 1;
		JactNext(jnew) = jnew;
		JactPrev(jnew) = jnew;
		return;
  		}

	j1 = j;
	jp = NULL;
	test = TRUE;
	len = JactLen( j);
	for ( i = 0; i < len; i++)
		{
		if ( test &&
				first_smaller( l, j1, jnew) && 
				first_smaller( l, jnew, JactNext( j1)))
			{
			test = FALSE;
			jp = j1;
			}
		JactLen( j1) += 1;
		j1 = JactNext( j1);
		}
	
	JactLen( jnew) = len+1;

	if ( !jp)
		{
		jp = JactPrev( j);
		}
	
	if ( jp == JactPrev( j) && first_smaller( l, jnew, j))
		{
		jactpoint[l] = jnew;
		}

	jact_join( jp, jnew, JactNext( jp));
}


static void
#if PROTO
	jactinsert(int l, int r)
#else
	jactinsert(l, r) int l, r;
#endif
{

	jactinsert1( l, r);
	jactinsert1( r, l);
}

static void
#if PROTO
	jactdelete1(int l, int r)
#else
	jactdelete1(l, r) int l, r;
#endif
{
	jact j, j1, jkill;
  	int i, len;
	
	j = jactpoint[ l];
  
  	assert( j);
  	len = JactLen( j);
	
	jkill = NULL;
	j1 = j;
	for ( i = 0; i < len; i++)
		{
		if ( JactElem( j1) == r)
			jkill = j1;
		JactLen( j1) -= 1;
		j1 = JactNext( j1);
  		}
	
	assert( jkill);

	JactJoin( JactPrev( jkill), JactNext( jkill));
	
	if ( jactpoint[l] == jkill)
		jactpoint[l] = JactNext( jkill);

  	des_jact( jkill);
}

static void
#if PROTO
  jactdelete(int l, int r)
#else
  jactdelete(l, r) int l, r;
#endif
{
	jactdelete1( l, r);
	jactdelete1( r, l);
}


static void
#if PROTO
	new_jact2( int a, int b)
#else
	new_jact2( a, b) int a, b;
#endif
{
  jact j1;
  
  assert( jactpoint[a] == NULL);
  
  j1 = cons_jact(b);

  JactLen( j1) = 1;
  JactNext(j1) = j1;	/* should be CCW  */
  JactPrev(j1) = j1;	/* not really important for two */
  JactAngle(j1) = fix_atan2( 
  			YCoord( b) - YCoord( a),
  			XCoord( b) - XCoord( a)
		     );

  jactpoint[a] = j1;
}

static void
#if PROTO
	new_jact( int a, int b, int c)
#else
	new_jact( a, b, c) int a, b, c;
#endif
{
  jact j1, j2;
  
  assert( jactpoint[a] == NULL);
  
  j1 = cons_jact(b);
  j2 = cons_jact(c);

  JactLen( j1) = 2;
  JactLen( j2) = 2;

  JactNext(j1) = j2;	/* should be CCW  */
  JactPrev(j1) = j2;	/* not really important for two */
  JactAngle(j1) = fix_atan2( 
  			YCoord( b) - YCoord( a),
  			XCoord( b) - XCoord( a)
		     );

  JactNext(j2) = j1;
  JactPrev(j2) = j1;
  JactAngle(j2) = fix_atan2( 
  			YCoord( c) - YCoord( a),
  			XCoord( c) - XCoord( a)
		     );

  if ( first_smaller( a, j1, j2))
	jactpoint[a] = j1;
  else
	jactpoint[a] = j2;
}


static int
#if PROTO
	qtest( int a, int b, int c, int d)
#else
	qtest( a, b, c, d) int a, b, c, d;
#endif
{
  double ax, ay, bx, by, cx, cy, dx, dy;
  double dr, di, nr, ni;
  
  if ( d == b)		/* in case we move right round */
  	return 1;
	
  ax = XCoord(a);
  bx = XCoord(b);
  cx = XCoord(c);
  dx = XCoord(d);
  ay = YCoord(a);
  by = YCoord(b);
  cy = YCoord(c);
  dy = YCoord(d);

  nr = -ax*bx + ay * by + ax * cx - 
		ay * cy + bx * dx - cx * dx - by * dy + cy * dy ;
  ni = -ay * bx - ax * by + ay * cx + 
		ax * cy + by * dx - cy * dx + bx * dy - cx * dy ;
  dr = ax * cx - bx * cx - ay * cy + 
		by * cy - ax * dx + bx * dx + ay * dy - by * dy ;
  di = ay * cx - by * cx + ax * cy - 
		bx * cy - ay * dx + by * dx - ax * dy + bx * dy ;

  if ( (dr*ni - di*nr) < EPS * (dr*dr + di*di))
  	return 1;

  return 0;
}



static hull
#if PROTO
	makeCCW( int a, int b, int c)
#else
	makeCCW( a, b, c) int a, b, c;
#endif
{
	int side;
  
	side = side_of_line( a, b, c);
  
	switch( side)
  		{
		case RIGHT:
			new_jact( c, b, a);
			new_jact( b, a, c);
			new_jact( a, c, b);
			return new_hull( a, c, b);
			
		case LEFT:
			new_jact( c, b, a);
			new_jact( b, a, c);
			new_jact( a, c, b);
			return new_hull( a, b, c);
			
		case COLLINEAR:
			new_jact2( a,b);	
			new_jact2( c,b);	
			new_jact( b,a,c);
			return new_hull4( a, b, c);

		default:
			assert( FALSE);
		}
	return NULL;
}

/*
 return CCW element from b in adjacency of a
*/
static int
#if PROTO
	succ(int a, int b)
#else
	succ(a, b) int a, b;
#endif
{
  jact j;
  
  j = jactpoint[ a];
  
  while (JactElem( j) != b) {
  	j = JactNext( j);
  }
  return JactElem( JactNext(j));
}

/*
 return CW element from b in adjacency of a
*/
static int
#if PROTO
	pred(int a, int b)
#else
	pred(a, b) int a, b;
#endif
{
  jact j;
  
  j = jactpoint[ a];
  
  while (JactElem( j) != b) {
  	j = JactNext( j);
  }
  
  return JactElem( JactPrev(j));
}


/*
 return the right-most point of h
*/
static hull
#if PROTO
	hright( hull h)
#else
	hright( h) hull h;
#endif
{
	int a, b;
	
	a = HullElem( h);
	b = HullElem( HullNext( h));
	
	if ( XCoord( b) > XCoord( a))
		return hright( HullNext( h));

	if ( XCoord( b) == XCoord( a))
		if ( YCoord( b) > YCoord( a))
			return hright( HullNext( h));
				
	b = HullElem( HullPrev( h));
	
	if ( XCoord( b) > XCoord( a))
		return hright( HullPrev( h));
	
	if ( XCoord( b) == XCoord( a))
		if ( YCoord( b) > YCoord( a))
			return hright( HullPrev( h));
				
	return h;
}

/*
 return the right-most point of h
*/
static hull
#if PROTO
	hleft( hull h)
#else
	hleft( h) hull h;
#endif
{
	int a, b;
	
	a = HullElem( h);
	b = HullElem( HullNext( h));

	if ( XCoord( b) < XCoord( a))
		return hleft( HullNext( h));
	
	if ( XCoord( b) == XCoord( a))
		if ( YCoord( b) < YCoord( a))
			return hleft( HullNext( h));	
	
	b = HullElem( HullPrev( h));
	if ( XCoord( b) < XCoord( a))
		return hleft( HullPrev( h));

	if ( XCoord( b) == XCoord( a))
		if ( YCoord( b) < YCoord( a))
			return hleft( HullPrev( h));	

	return h;
}


static hull
#if PROTO
	addpoint_hull_low( hull x, int a)
#else
	addpoint_hull_low( x, a) hull x; int a;
#endif
{
	int side;
	hull xn;
	
	xn = HullPrev( x);
	
	side = side_of_line( a, HullElem(x), HullElem(xn));
	
	if ( side == LEFT)
		return addpoint_hull_low( xn, a); /* go CW */
	
	return x;
}

static hull
#if PROTO
	addpoint_hull_upp( hull x, int a)
#else
	addpoint_hull_upp( x, a) hull x; int a;
#endif
{
	int side;
	hull xn;
	
	xn = HullNext( x);

	side = side_of_line( a, HullElem(x), HullElem(xn));
	
	if ( side == RIGHT)
		return addpoint_hull_upp( xn, a); /* go CW */

	return x;
}


static void 
#if PROTO
	addpoint_fun( int l, int u, int r)
#else
	addpoint_fun( l, u, r) int l, u, r;
#endif
{
	int l1, l2;
  
	jactinsert( l, r);
	if ( l == u)
		return;

	l1 = succ( l, r);
	l2 = succ( l, l1);

	while( !qtest( l, r, l1, l2)) 
		{
		jactdelete( l, l1);
		l1 = l2;
		l2 = succ( l, l1);
		}

	addpoint_fun( l1, u, r);
}


/* Join two hulls.

   Points on the lefthull between llow and lupp
   and on the right hull between rlow and rupp
   will be discarded.

   Care needs to be taken if it is a linear hull
   and if the end and start points are the same.
   
   At the end collinear points are removed.
*/
static hull 
#if PROTO
	hull_join( hull llow, hull lupp, hull rlow, hull rupp)
#else
	hull_join( llow, lupp, rlow, rupp) hull llow, lupp, rlow, rupp;
#endif
{
	hull h, ho;
	
	h = HullNext( llow);
	
	while ( h != lupp)
		{
		ho = h;
		h = HullNext( h);
		des_hull( ho);
		}

	h = HullNext( rupp);
	
	while ( h != rlow)
		{
		ho = h;
		h = HullNext( h);
		des_hull( ho);
		}

	HullJoin( llow, rlow);
	HullJoin( rupp, lupp);

	return llow;
}


static void
#if PROTO
	clear_collinear( hull h)
#else
	clear_collinear( h) hull h;
#endif
{
	hull hl;
	
	HullCollinear( h) = FALSE;
	hl = HullNext( h);
	while( hl != h)
		{
		HullCollinear( hl) = FALSE;
		hl = HullNext( hl);
		}
}

static void
#if PROTO
	fix_collinear_hull( hull h, hull *hl, hull *hu)
#else
	fix_collinear_hull( h, hl, hu) hull h, *hl, *hu;
#endif
{
	hull h1, h2;
	
	h1 = cons_hull( HullElem( h));
	HullCollinear( h1) = TRUE;
	h2 = HullNext( h);
			
	HullJoin( h, h1);
	HullJoin( h1, h2);
			
	*hu = h1;
	*hl = h;
}


static hull 
#if PROTO
	addpoint( hull h, int a)
#else
	addpoint( h, a) hull h; int a;
#endif
{
	hull hl, hu, r;
	int side, flag;

	flag = TRUE;
	r = hright( h);
	
	if ( HullCollinear( r))
		{
		side = 
		  side_of_line( HullElem( HullNext( r)),
		  				HullElem( r), a);
		if ( side == COLLINEAR)
			{
			flag = FALSE;
			fix_collinear_hull( r, &hl, &hu);
			}
		else
			{
			clear_collinear( h);
			}
		}
		
	if ( flag)
		{
		hl = addpoint_hull_low( r, a);
		hu = addpoint_hull_upp( r, a);
		}
		
	addpoint_fun( HullElem( hl), HullElem( hu), a);

	r = cons_hull( a);
	if ( !flag)
		HullCollinear( r) = TRUE;
		
	HullJoin( r, r);

	return hull_join( hl, hu, r, r);
}

static hull 
#if PROTO
	simple_delaunay( int a, int b)
#else
	simple_delaunay( a, b) int a, b;
#endif
{
  int len;
  hull h;
  
  h = makeCCW( a, a+1, a+2);

  len = b-a+1;
  assert( len > 2 && len < 6);
  
  if (len > 3)
  	h = addpoint( h, a+3);
  if (len > 4)
	h = addpoint ( h, a+4);
  return h;
}

static void 
#if PROTO
	merge_delaunay_fun( int l, int lupp, int r, int rupp)
#else
	merge_delaunay_fun( l, lupp, r, rupp) int l, lupp, r, rupp;
#endif
{
  int a, b, r1, r2, l1, l2;

  
  jactinsert( l, r);
  if ( l == lupp && r == rupp)
  	return;
	
  a = FALSE;
  b = FALSE;
  
  r1 = pred( r, l);
  if (right_of_line( r, l, r1)) {
  	r2 = pred( r, r1);
	while( !qtest( r1, l, r, r2) && right_of_line( r, l, r2)) {
		jactdelete( r, r1);
		r1 = r2;
		r2 = pred( r, r1);
	}
  } else
  	a = TRUE;

  l1 = succ( l, r);
  if (right_of_line( r, l, l1)) {
  	l2 = succ( l, l1);
	while( !qtest( l, r, l1, l2) && right_of_line( r, l, l2)) {
		jactdelete( l, l1);
		l1 = l2;
		l2 = succ( l, l1);
	}
  } else
  	b = TRUE;
  
  if ( a) {
  	l = l1;
  } else {
  	if ( b) {
		r = r1;
	} else {
		if (qtest( l, r, r1, l1)) {
			r = r1;
		} else {
			l = l1;
		}
	}
  }
  merge_delaunay_fun( l, lupp, r, rupp);
}

static int 
#if PROTO
	joinhull_low( hull l, hull r, hull * la, hull * ra)
#else
	joinhull_low( l, r, la, ra) hull l, r, * la, * ra;
#endif
{
	hull nl, nr;
	int side;
/*
  is it r or nr ?
*/
	nr = HullNext( r);
 	side = side_of_line( HullElem( l), HullElem( r), HullElem( nr));

	if ( side == RIGHT)
		return joinhull_low( l, nr, la, ra);

	if ( side == COLLINEAR)
		{
		if ( dist2( HullElem( l), HullElem( r)) > 
		     dist2( HullElem( l), HullElem( nr)))
			return joinhull_low( l, nr, la, ra);
		}

/*
  is it l or nl ?
*/
	nl = HullPrev( l);
 	side = side_of_line( HullElem( l), HullElem( r), HullElem( nl));

	if ( side == RIGHT)
		return joinhull_low( nl, r, la, ra);

	if ( side == COLLINEAR)
		{
		if ( dist2( HullElem( r), HullElem( l)) > 
		     dist2( HullElem( r), HullElem( nl)))
			return joinhull_low( nl, r, la, ra);
		}
		
	*la = l;
	*ra = r;
	return TRUE;
}

static int 
#if PROTO
	joinhull_upp( hull l, hull r, hull * la, hull * ra)
#else
	joinhull_upp( l, r, la, ra) hull l, r, * la, * ra;
#endif
{
	hull nl, nr;
	int side;
  
/*
  is it r or nr ?
*/
	nr = HullPrev( r);
 	side = side_of_line( HullElem( r), HullElem( l), HullElem( nr));

	if ( side == RIGHT)
		return joinhull_upp( l, nr, la, ra);
	
	if ( side == COLLINEAR)
		{
		if ( dist2( HullElem( l), HullElem( r)) > 
		     dist2( HullElem( l), HullElem( nr)))
			return joinhull_low( l, nr, la, ra);
		}

/*
  is it l or nl ?
*/
	nl = HullNext( l);
 	side = side_of_line( HullElem( r), HullElem( l), HullElem( nl));

	if ( side == RIGHT)
		return joinhull_upp( nl, r, la, ra);

	if ( side == COLLINEAR)
		{
		if ( dist2( HullElem( r), HullElem( l)) > 
		     dist2( HullElem( r), HullElem( nl)))
			return joinhull_low( nl, r, la, ra);
		}

	*la = l;
	*ra = r;
	return TRUE;
}

static hull 
#if PROTO
	merge_delaunay( hull hl, hull hr)
#else
	merge_delaunay( hl, hr) hull hl, hr;
#endif
{
	hull l, r, llow, rlow, lupp, rupp;
	int side, flagl, flagr;
	
	l = hright( hl);
	r = hleft( hr);

	flagl = flagr = FALSE;
	if ( HullCollinear( l))		
		{
		side = 
		  side_of_line( HullElem( HullNext( l)),
		  				HullElem( l), HullElem( r));
		if ( side == COLLINEAR)
			flagl = TRUE;
		}
		
	if ( HullCollinear( r))		
		{
		side = 
		  side_of_line( HullElem( HullNext( r)),
		  				HullElem( r), HullElem( l));
		if ( side == COLLINEAR)
			flagr = TRUE;

		}

	if ( flagl && flagr)
		{
		fix_collinear_hull( l, &llow, &lupp);
		fix_collinear_hull( r, &rupp, &rlow);
		}
	else
		{
		if ( flagl)
			clear_collinear( l);
		if ( flagr)
			clear_collinear( r);
		(void) joinhull_low( l, r, &llow, &rlow);
		(void) joinhull_upp( l, r, &lupp, &rupp);
		}

	merge_delaunay_fun( 
  		HullElem(llow), HullElem(lupp), 
  		HullElem(rlow), HullElem(rupp));

	return hull_join( llow, lupp, rlow, rupp);
}

static hull 
#if PROTO
	delaunay_fun( int a, int b)
#else
	delaunay_fun( a, b) int a, b;
#endif
{
  int len;

  len = b-a+1;
  if ( len > 5)
  	return
  	    merge_delaunay(
		delaunay_fun( a, a+(int) floor(len/2.0)-1),
		delaunay_fun( a+ (int) floor(len/2.0), b)) ;
  else
  	return
	    simple_delaunay( a, b);
}



#define SetTri( t, i, p2, p3)	\
		(TriA(t) = Encode( i)+1, \
		 TriB(t) = Encode( p2)+1, \
		 TriC(t) = Encode( p3)+1)


static void 
#if PROTO
 	write_triangles( int len, hull h)
#else
 	write_triangles( len, h) int len; hull h;
#endif
{
  	int p1, pp, pn, ntri, side;
  	tri tres, t1, tn;
	jact j1, jn, jp;
	
	ntri = 0;
  	tn = tres = cons_tri();

	for ( p1 = 0; p1 < len; p1++)
		{
		if ( TriUsed( p1) < TriNum( p1))
			{
			jn = j1 = jactpoint[ p1];
			do
				{
				jp = jn;
				jn = JactNext( jn);			
				pp = JactElem( jp);
				pn = JactElem( jn);
				side = side_of_line( p1, pp, pn);
				if ( side == LEFT &&
						TriUsed( p1) < TriNum( p1) &&
 						TriUsed( pp) < TriNum( pp) &&
				  		TriUsed( pn) < TriNum( pn))
					{
					TriUsed( p1)++;
					TriUsed( pn)++;
					TriUsed( pp)++;
					SetTri( tn, p1, pp, pn); 
					t1 = cons_tri();
					TriNext( tn) = t1;
					tn = t1;
					ntri++;
					}
				}
			while ( jn != j1);
			}
		assert( TriUsed( p1) == TriNum( p1));
		}

  	tn = tres;
  	MLPutFunction( stdlink, "List", ntri);
  	while ( TriNext(tn) != NULL) 
		{
  		MLPutFunction( stdlink, "List", 3);
		MLPutInteger( stdlink, TriA( tn));
		MLPutInteger( stdlink, TriB( tn));
		MLPutInteger( stdlink, TriC( tn));
		t1 = tn;
		tn = TriNext(t1);
		des_tri( t1);
  		}
  	des_tri( TriNext(tn));
  	des_tri( tn);
}
		



void
#if PROTO
	delaunay( void)
#else
	delaunay()
#endif
{
	long lenl, tmp;
	int i, len, hlen;
	hull hres, hn;
	
	if (!MLCheckFunction( stdlink, "List", &lenl))
		goto error;
	len = lenl;
	dlen = len;
	cons_data( len);
	for ( i = 0; i < len; i++) 
		{
  		if ( ! MLCheckFunction( stdlink, "List", &tmp))
			goto error;
		if ( tmp != 2)
			goto error;
		if (!MLGetDouble( stdlink, &pnts[i].x) ||
	    	!MLGetDouble( stdlink, &pnts[i].y))
	    	goto error;
  		}
		
  	qsort( &pnts[0], len, sizeof( spoint), ordered); 

  	hres = delaunay_fun( 0, len-1);

	MLPutFunction( stdlink, "List", 2);
	hlen = len_hull( hres);
	MLPutFunction( stdlink, "List", hlen);
	for ( i = 0; i < hlen; i++) 
		{
		MLPutInteger( stdlink, Encode( HullElem( hres))+1);
		hres = HullNext( hres);
		}

	hn = HullNext( hres);
	HullPt( HullElem( hres)) = TRUE;
	while( hn != hres)
		{
		HullPt( HullElem( hn)) = TRUE;
		hn = HullNext( hn);
		}
	
	for ( i = 0; i < len; i++)
		{
		TriNum( i) = JactLen( jactpoint[i]);
		if ( HullPt( i))
			TriNum( i)--;
		}


	write_triangles( len, hres);

	des_data( hres, len);
	return;
 
error:
	des_data( hres, len);
	MLPutSymbol( stdlink, "$Failed");
	return;
}


#if !WINDOWS_MATHLINK

int main(argc, argv)
        int argc; char* argv[];
{
        return MLMain(argc, argv);
}

#else

int PASCAL WinMain( HANDLE hinstCurrent, HANDLE hinstPrevious, LPSTR lpszCmdLine, int nCmdShow)
{
	char  buff[512];
	char FAR * argv[32];
	int argc;

	if( !MLInitializeIcon( hinstCurrent, nCmdShow)) return 1;
	argc = MLStringToArgv( lpszCmdLine, buff, argv, 32);
	return MLMain( argc, argv);
}
#endif
