// Copyright (c) FIRST and other WPILib contributors.
// Open Source Software; you can modify and/or share it under the terms of
// the WPILib BSD license file in the root directory of this project.

#include "frc/SpeedControllerGroup.h"

#include <wpi/sendable/SendableBuilder.h>
#include <wpi/sendable/SendableRegistry.h>

using namespace frc;

// Can't use a delegated constructor here because of an MSVC bug.
// https://developercommunity.visualstudio.com/content/problem/583/compiler-bug-with-delegating-a-constructor.html

SpeedControllerGroup::SpeedControllerGroup(
    std::vector<std::reference_wrapper<SpeedController>>&& speedControllers)
    : m_speedControllers(std::move(speedControllers)) {
  Initialize();
}

void SpeedControllerGroup::Initialize() {
  for (auto& speedController : m_speedControllers) {
    wpi::SendableRegistry::AddChild(this, &speedController.get());
  }
  static int instances = 0;
  ++instances;
  wpi::SendableRegistry::Add(this, "SpeedControllerGroup", instances);
}

void SpeedControllerGroup::Set(double speed) {
  for (auto speedController : m_speedControllers) {
    speedController.get().Set(m_isInverted ? -speed : speed);
  }
}

double SpeedControllerGroup::Get() const {
  if (!m_speedControllers.empty()) {
    return m_speedControllers.front().get().Get() * (m_isInverted ? -1 : 1);
  }
  return 0.0;
}

void SpeedControllerGroup::SetInverted(bool isInverted) {
  m_isInverted = isInverted;
}

bool SpeedControllerGroup::GetInverted() const {
  return m_isInverted;
}

void SpeedControllerGroup::Disable() {
  for (auto speedController : m_speedControllers) {
    speedController.get().Disable();
  }
}

void SpeedControllerGroup::StopMotor() {
  for (auto speedController : m_speedControllers) {
    speedController.get().StopMotor();
  }
}

void SpeedControllerGroup::InitSendable(wpi::SendableBuilder& builder) {
  builder.SetSmartDashboardType("Speed Controller");
  builder.SetActuator(true);
  builder.SetSafeState([=] { StopMotor(); });
  builder.AddDoubleProperty(
      "Value", [=] { return Get(); }, [=](double value) { Set(value); });
}
